// Emacs style mode select	 -*- C++ -*- 
//-----------------------------------------------------------------------------
//
// $Id: d_main.cpp,v 1.9 2004/10/16 18:45:49 incubus Exp $
//
// Copyright (C) 1993-1996 by id Software, Inc.
//
// This source is available for distribution and/or modification
// only under the terms of the DOOM Source Code License as
// published by id Software. All rights reserved.
//
// The source is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// FITNESS FOR A PARTICULAR PURPOSE. See the DOOM Source Code License
// for more details.
//
// $Log: d_main.cpp,v $
// Revision 1.9  2004/10/16 18:45:49  incubus
// More savegame removal
//
// Revision 1.8  2004/10/13 02:34:16  incubus
// Minor modifications so the OS information shows up in the launcher if you're chrooted
//
// Revision 1.7  2004/10/12 22:09:13  incubus
// Moved the watchdog section so it started after the chroot (don't want the thread to have root privledges, do we?)
//
// Revision 1.6  2004/10/12 05:18:18  incubus
// Added the capability for zserv to chroot itself.
//
// Revision 1.5  2004/10/10 18:40:27  incubus
// Sync with ZDaemon 1.06.07
//
// Revision 1.4  2004/07/23 09:25:43  incubus
// Removal of some unused or unnecessary functions.
//
// Revision 1.3  2004/07/21 22:24:20  incubus
// Remove StatusBar code dependancies. It's not needed in the server.
//
// Revision 1.2  2004/07/21 04:55:39  incubus
// Sync with ZDaemon 1.06.04b
//
// Revision 1.1.1.1  2004/07/20 02:19:20  incubus
// ZDaemon 1.06 source import
//
//
// DESCRIPTION:
//		DOOM main program (D_DoomMain) and game loop (D_DoomLoop),
//		plus functions to determine game mode (shareware, registered),
//		parse command line parameters, configure game parameters (turbo),
//		and call the startup functions.
//
//-----------------------------------------------------------------------------

// HEADER FILES ------------------------------------------------------------

#ifdef _WIN32

//Raider added this to make GetModuleFileName work:
#include <windows.h>

#include <direct.h>
#define mkdir(a,b) _mkdir (a)
#else
#include <sys/stat.h>
#endif

#ifdef unix
#include <sys/types.h> // [Dash|RD] Needed for getuid, which is used for the chroot section.
#include <unistd.h>
#include <version.h> // [Dash|RD] Needed for the SV_SETPLATFORM_ID() function.
#endif

#include <time.h>
#include <math.h>

#include "errors.h"

#include "d_gui.h"
#include "m_alloc.h"
#include "m_random.h"
#include "doomdef.h"
#include "doomstat.h"
#include "gstrings.h"
#include "z_zone.h"
#include "w_wad.h"
#include "s_sound.h"
#include "v_video.h"
#include "f_finale.h"
#include "f_wipe.h"
#include "m_argv.h"
#include "m_misc.h"
#include "c_console.h"
#include "c_dispatch.h"
#include "i_system.h"
//#include "i_sound.h"
#include "i_video.h"
#include "g_game.h"
#include "hu_stuff.h"
#include "wi_stuff.h"
#include "st_stuff.h"
#include "am_map.h"
#include "p_setup.h"
#include "r_local.h"
#include "r_sky.h"
#include "d_main.h"
#include "d_dehacked.h"
#include "cmdlib.h"
#include "s_sound.h"
#include "m_swap.h"
#include "v_text.h"
#include "gi.h"
#include "b_bot.h"		//Added by MC:
#include "stats.h"
#include "a_doomglobal.h"
#include "gameconfigfile.h"
#include "decallib.h"
#include "v_text.h"
#include "sv_main.h"
#include "log.h"
#include "watchdog.h"

int		zd_countwad;

#ifdef unix
char		Platform_ID[255];
#endif

// MACROS ------------------------------------------------------------------

// TYPES -------------------------------------------------------------------

struct GameAtExit
{
	GameAtExit *Next;
	char Command[1];
};

// EXTERNAL FUNCTION PROTOTYPES --------------------------------------------

extern void R_ExecuteSetViewSize ();
extern void G_NewInit ();

// PUBLIC FUNCTION PROTOTYPES ----------------------------------------------

void D_CheckNetGame ();
void D_AddFile (const char *file);

// PRIVATE FUNCTION PROTOTYPES ---------------------------------------------

static const char *BaseFileSearch (const char *file, const char *ext);

// EXTERNAL DATA DECLARATIONS ----------------------------------------------

EXTERN_CVAR (Float, turbo)
EXTERN_CVAR (Int, crosshair)
EXTERN_CVAR (Int, force_password)		//Bond
EXTERN_CVAR (String, password)			//Bond
EXTERN_CVAR (String, optional_wads)
EXTERN_CVAR (Int,	master_advertise)
EXTERN_CVAR (Int,   removebotswhenhumans)	//Kilgore


extern gameinfo_t SharewareGameInfo;
extern gameinfo_t RegisteredGameInfo;
extern gameinfo_t RetailGameInfo;
extern gameinfo_t CommercialGameInfo;
extern gameinfo_t HereticGameInfo;
extern gameinfo_t HereticSWGameInfo;
extern gameinfo_t HexenGameInfo;

extern BOOL netdemo;
extern int NewWidth, NewHeight, NewBits, DisplayBits;
EXTERN_CVAR (Bool, st_scale)
extern BOOL gameisdead;

extern cycle_t WallCycles, PlaneCycles, MaskedCycles, WallScanCycles;

// PUBLIC DATA DEFINITIONS -------------------------------------------------

// [NightFang] - both fraglimit and timelimit are archived now
CVAR (Int, fraglimit, 0, CVAR_ARCHIVE|CVAR_SERVERINFO);
CVAR (Int, timelimit, 0, CVAR_ARCHIVE|CVAR_SERVERINFO);
CVAR (Bool, queryiwad, true, CVAR_ARCHIVE|CVAR_GLOBALCONFIG);
CVAR (Int, teamscorelimit, 0, CVAR_ARCHIVE|CVAR_SERVERINFO);	//Kilgore
CVAR (Int, maxteams, 4, CVAR_ARCHIVE|CVAR_SERVERINFO);			//Kilgore
CVAR (Int, spam_window, 10, CVAR_ARCHIVE|CVAR_SERVERINFO);		//Kilgore
CVAR (Int, spam_limit,  3, CVAR_ARCHIVE|CVAR_SERVERINFO);		//Kilgore
CVAR (Bool, sv_chroot, false, CVAR_ARCHIVE|CVAR_GLOBALCONFIG); 		       	// [Dash|RD]
CVAR (String, sv_chrootjail, "./zserv_jail/", CVAR_ARCHIVE|CVAR_GLOBALCONFIG); 	// [Dash|RD]
CVAR (Int, sv_chrootuid, 65534, CVAR_ARCHIVE|CVAR_GLOBALCONFIG);	    	// [Dash|RD]
EXTERN_CVAR (Int,	minplayers)			//Kilgore

EXTERN_CVAR (Int, load_skins)


bool DrawFSHUD;				// [RH] Draw fullscreen HUD?
wadlist_t *wadfiles;		// [RH] remove limit on # of loaded wads
BOOL devparm;				// started game with -devparm
char *D_DrawIcon;			// [RH] Patch name of icon to draw on next refresh
int NoWipe;					// [RH] Allow wipe? (Needs to be set each time)
BOOL singletics = false;	// debug flag to cancel adaptiveness
char startmap[8];
BOOL autostart;
FILE *debugfile;
event_t events[MAXEVENTS];
int eventhead;
int eventtail;
gamestate_t wipegamestate = GS_DEMOSCREEN;	// can be -1 to force a wipe
DCanvas *page;

cycle_t FrameCycles;

const char *IWADTypeNames[NUM_IWAD_TYPES] =
{
	"DOOM 2: TNT - Evilution",
	"DOOM 2: Plutonia Experiment",
	"Hexen: Beyond Heretic",
	"DOOM 2: Hell on Earth",
	"Heretic Shareware",
	"Heretic: Shadow of the Serpent Riders",
	"Heretic",
	"DOOM Shareware",
	"The Ultimate DOOM",
	"DOOM Registered"
};

// PRIVATE DATA DEFINITIONS ------------------------------------------------

static wadlist_t **wadtail = &wadfiles;
static int demosequence;
static const char *IWADNames[] =
{
	NULL,
	"doom2f.wad",
	"doom2.wad",
	"plutonia.wad",
	"tnt.wad",
	"doomu.wad", // Hack from original Linux version. Not necessary, but I threw it in anyway.
	"doom.wad",
	"doom1.wad",
	"heretic.wad",
	"heretic1.wad",
	"hexen.wad",
	NULL
};
static GameAtExit *ExitCmdList;

// CODE --------------------------------------------------------------------

//==========================================================================
//
// D_PostEvent
//
// Called by the I/O functions when input is detected.
//
//==========================================================================

void D_PostEvent (const event_t *ev)
{
	events[eventhead] = *ev;
	eventhead = (++eventhead)&(MAXEVENTS-1);
}

//==========================================================================
//
// CVAR dmflags
//
//==========================================================================
CVAR (Int,	dmflags,			0,		 CVAR_SERVERINFO|CVAR_ARCHIVE);
CVAR (Flag, sv_nohealth,		dmflags, DF_NO_HEALTH);
CVAR (Flag, sv_noitems,			dmflags, DF_NO_ITEMS);
CVAR (Flag, sv_weaponstay,		dmflags, DF_WEAPONS_STAY);
CVAR (Flag, sv_falldamage,		dmflags, DF_FORCE_FALLINGHX);
CVAR (Flag, sv_oldfalldamage,	dmflags, DF_FORCE_FALLINGZD);
CVAR (Flag, sv_samelevel,		dmflags, DF_SAME_LEVEL);
CVAR (Flag, sv_spawnfarthest,	dmflags, DF_SPAWN_FARTHEST);
CVAR (Flag, sv_forcerespawn,	dmflags, DF_FORCE_RESPAWN);
CVAR (Flag, sv_noarmor,			dmflags, DF_NO_ARMOR);
CVAR (Flag, sv_noexit,			dmflags, DF_NO_EXIT);
CVAR (Flag, sv_infiniteammo,	dmflags, DF_INFINITE_AMMO);
CVAR (Flag, sv_nomonsters,		dmflags, DF_NO_MONSTERS);
CVAR (Flag, sv_monsterrespawn,	dmflags, DF_MONSTERS_RESPAWN);
CVAR (Flag, sv_itemrespawn,		dmflags, DF_ITEMS_RESPAWN);
CVAR (Flag, sv_fastmonsters,	dmflags, DF_FAST_MONSTERS);
CVAR (Flag, sv_nojump,			dmflags, DF_NO_JUMP);
CVAR (Flag, sv_nofreelook,		dmflags, DF_NO_FREELOOK);
CVAR (Flag, sv_respawnsuper,	dmflags, DF_RESPAWN_SUPER);
CVAR (Flag, sv_nopassover,		dmflags, DF_NO_PASSMOBJ);
CVAR (Flag, sv_nofov,			dmflags, DF_NO_FOV);

//==========================================================================
//
// CVAR dmflags2
//
// [RH] From Skull Tag. Some of these were already done as separate cvars
// (such as bfgaiming), but I collected them here like Skull Tag does.
//
//==========================================================================

// [NightFang] - added the archive flag
CVAR (Int, dmflags2, 1024, CVAR_SERVERINFO | CVAR_ARCHIVE);
CVAR (Flag, sv_weapondrop,		dmflags2, DF2_YES_WEAPONDROP);
// [NightFang]- enabled by default
CVAR (Flag, sv_respawnprotect,	dmflags2, DF2_YES_INVUL);
CVAR (Flag, sv_barrelrespawn,	dmflags2, DF2_BARRELS_RESPAWN);
// [NightFang] - Newest stuff for zdaemon
CVAR (Flag, sv_samespawnspot,	dmflags2, DF2_SAME_SPAWN_SPOT);
CVAR (Flag, sv_niceweapons,		dmflags2, DF2_NICE_WEAPONS);
CVAR (Flag, sv_keepkeys,		dmflags2, DF2_KEEP_KEYS);
CVAR (Flag, sv_oldwepsound,		dmflags2, DF2_OLDWEPSOUND);

//==========================================================================
//
// D_Display
//
// Draw current display, possibly wiping it from the previous
//
//==========================================================================

void D_Display (bool screenshot)
{
}

//==========================================================================
//
// DoConsoleAtExit
//
// Executes the contents of the atexit cvar, if any, at quit time.
//
//==========================================================================

static void STACK_ARGS DoConsoleAtExit ()
{
	GameAtExit *cmd = ExitCmdList;

	while (cmd != NULL)
	{
		GameAtExit *next = cmd->Next;
		AddCommandString (cmd->Command);
		Z_Free (cmd);
		cmd = next;
	}
	ExitCmdList = NULL;
}

//==========================================================================
//
// CCMD atexit
//
//==========================================================================

CCMD (atexit)
{
	if (argv.argc() == 1)
	{
		Printf ("Registered atexit commands:\n");
		GameAtExit *record = ExitCmdList;
		while (record != NULL)
		{
			Printf ("%s\n", record->Command);
			record = record->Next;
		}
		return;
	}
	for (int i = 1; i < argv.argc(); ++i)
	{
		GameAtExit *record = (GameAtExit *)Z_Malloc (
			sizeof(GameAtExit)+strlen(argv[i]), PU_STATIC, 0);
		strcpy (record->Command, argv[i]);
		record->Next = ExitCmdList;
		ExitCmdList = record;
	}
}

//==========================================================================
//
// D_ErrorCleanup ()
//
// Cleanup after a recoverable error.
//==========================================================================

void D_ErrorCleanup ()
{
	// [NightFang] - removed
	//screen->Unlock ();
	bglobal.RemoveAllBots (true);
	D_QuitNetGame ();
	Net_ClearBuffers ();
	G_NewInit ();
	singletics = false;
	playeringame[0] = 1;
	players[0].playerstate = PST_LIVE;
	gameaction = ga_fullconsole;
}



/*
 * Kilgore: Do some extra initializations and simple sanity checks
 */
static void parameter_checks(void)
{
	int			v;
	UCVarValue	num;

	switching_kill = false;

	v = timelimit;
	if (v<0)
	{
		num.Int = 0;
		timelimit.ForceSet(num,CVAR_Int);
	}

	v = maxteams;
	if (v<2 || v>NUM_TEAMS)
	{
		num.Int = NUM_TEAMS;
		maxteams.ForceSet(num,CVAR_Int);
	}

	#if __CTF__
	//duke
	//==
	if (ctf)
	{
		num.Int = 1;	teamplay.ForceSet(num,CVAR_Int);
		num.Int = 2;	maxteams.ForceSet(num,CVAR_Int);
		num.Int = 0;	fraglimit.ForceSet(num,CVAR_Int);
	}
	//==
	#endif

	if (teamplay)
	{
		num.Int = 1;
		deathmatch.ForceSet(num,CVAR_Int);
	}

	v = spam_window;
	if (v<0 || v>1000)
	{
		num.Int = 0;
		spam_window.ForceSet(num,CVAR_Int);
	}

	v = spam_limit;
	if (v<0 || v>1000)
	{
		num.Int = 0;
		spam_limit.ForceSet(num,CVAR_Int);
	}

	if (force_password)							// Bond: check for empty password
	{
		UCVarValue  pw = password.GetGenericRep(CVAR_String);
		if (strlen(pw.String)==0)
		{
			num.Int = 0;
			force_password.ForceSet(num,CVAR_Int);
		}
	}

	int maxcl, maxpl, minpl;

	maxcl = maxclients;
	maxpl = maxplayers;
	minpl = minplayers;

	if (maxcl<2 || maxcl>MAXPLAYERS)	maxcl = MAXPLAYERS;
	if (maxpl<2 || maxpl>maxcl)			maxpl = maxcl;

	if (minpl<2)						minpl = 0;
	else if (minpl>maxpl)				minpl = maxpl;

	if (minpl>0 && maxpl==maxcl)
	{
		if (maxcl<MAXPLAYERS)
		{
			maxcl++;
		}
		else
		{
			maxpl = maxcl-1;
			if (minpl>maxpl) minpl = maxpl;
		}
	}

	num.Int = maxcl;		maxclients.ForceSet(num,CVAR_Int);
	num.Int = maxpl;		maxplayers.ForceSet(num,CVAR_Int);
	num.Int = minpl;		minplayers.ForceSet(num,CVAR_Int);

	if (minpl>0)
		bglobal.wanted_botnum = 0;

	if (master_advertise != 0)
	{
		num.Int = 1;
		master_advertise.ForceSet(num,CVAR_Int);
		enable_conn_weap_logs();
		num.Int = 10;
		spam_window.ForceSet(num,CVAR_Int);
		num.Int = 3;
		spam_limit.ForceSet(num,CVAR_Int);
	}

	if (removebotswhenhumans != 0)
	{
		num.Int = 1;
		removebotswhenhumans.ForceSet(num,CVAR_Int);
	}

	// Set optional wads
	UCVarValue wl = optional_wads.GetGenericRep(CVAR_String);
	WC_set_optional(wl.String);

}

//==========================================================================
//
// D_DoomLoop
//
// Manages timing and IO, calls all ?_Responder, ?_Ticker, and ?_Drawer,
// calls I_GetTime, I_StartFrame, and I_StartTic
//
//==========================================================================
void D_DoomLoop ()
{
	for (;;)
	{
		try
		{
			SV_RunTics();
		}
		catch (CRecoverableError &error)
		{
			if (error.GetMessage ())
				Printf_Bold ("\n%s\n", error.GetMessage());
			D_ErrorCleanup ();
		}
	}
}

//==========================================================================
//
// D_StartTitle
//
//==========================================================================

void D_StartTitle (void)
{
	gameaction = ga_nothing;
	demosequence = -1;
}

//==========================================================================
//
// Cmd_Endgame
//
// [RH] Quit the current game and go to fullscreen console
//
//==========================================================================

CCMD (endgame)
{
	if (!netgame)
	{
		gameaction = ga_fullconsole;
		demosequence = -1;
	}
}

//==========================================================================
//
// D_AddFile
//
//==========================================================================

void D_AddFile (const char *file)
{
	if (file == NULL)
	{
		return;
	}

	if (!FileExists (file))
	{
		const char *f = BaseFileSearch (file, ".wad");
		if (f == NULL)
		{
			Printf ("Can't find '%s'\n", file);
			return;
		}
		file = f;
	}
	wadlist_t *wad = (wadlist_t *)Z_Malloc (sizeof(*wad) + strlen(file), PU_STATIC, 0);

	*wadtail = wad;
	wad->next = NULL;
	
	strcpy (wad->name, file);
	wadtail = &wad->next;
}

//==========================================================================
//
// D_AddConfigWads
//
// Adds all files in the specified config file section.
//
//==========================================================================

void D_AddConfigWads (const char *section)
{
	if (GameConfig->SetSection (section))
	{
		const char *key;
		const char *value;
		FConfigFile::Position pos;

		while (GameConfig->NextInSection (key, value))
		{
			if (stricmp (key, "Path") == 0)
			{
				// BaseFileSearch resets GameConfig's position, so remember it
				GameConfig->GetPosition (pos);

				const char *wadfile = BaseFileSearch (value, "wad");

				if (wadfile != NULL)
				{
					D_AddFile (wadfile);
				}
				else
				{ // Try pattern matching
					findstate_t findstate;
					long handle = I_FindFirst (value, &findstate);

					 if (handle != -1)
					 {
						 do
						 {
							 D_AddFile (I_FindName (&findstate));
						 } while (I_FindNext (handle, &findstate) == 0);
					 }
					 I_FindClose (handle);
				}

				// Reset GameConfig's position to get next wad
				GameConfig->SetPosition (pos);
			}
		}
	}
}

//==========================================================================
//
// D_AddDirectory
//
// Add all .wad files in a directory. Does not descend into subdirectories.
//
//==========================================================================

static void D_AddDirectory (const char *dir)
{
	char curdir[PATH_MAX];

	if (getcwd (curdir, PATH_MAX))
	{
		char skindir[PATH_MAX];
		findstate_t findstate;
		long handle;
		int stuffstart;

		stuffstart = strlen (dir);
		memcpy (skindir, dir, stuffstart*sizeof(*dir));
		skindir[stuffstart] = 0;

		if (skindir[stuffstart-1] == '/')
		{
			skindir[--stuffstart] = 0;
		}

		if (!chdir (skindir))
		{
			skindir[stuffstart++] = '/';
			if ((handle = I_FindFirst ("*.wad", &findstate)) != -1)
			{
				do
				{
					if (!(I_FindAttr (&findstate) & FA_DIREC))
					{
						strcpy (skindir + stuffstart,
								I_FindName (&findstate));
						D_AddFile (skindir);
						//Printf("Raider Debug: skindir=>%s<=\n",skindir);
					}
				} while (I_FindNext (handle, &findstate) == 0);
				I_FindClose (handle);
			}
		}
		chdir (curdir);
	}
}

//==========================================================================
//
// SetIWAD
//
// Sets parameters for the game using the specified IWAD.
//==========================================================================

static void SetIWAD (const char *iwadpath, EIWADType type)
{
	static const struct
	{
		GameMode_t Mode;
		const gameinfo_t *Info;
		GameMission_t Mission;
	} Datas[NUM_IWAD_TYPES] = {
		{ commercial,	&CommercialGameInfo,	pack_tnt },		// Doom2TNT
		{ commercial,	&CommercialGameInfo,	pack_plut },	// Doom2Plutonia
		{ commercial,	&HexenGameInfo,			doom2 },		// Hexen
		{ commercial,	&CommercialGameInfo,	doom2 },		// Doom2
		{ shareware,	&HereticSWGameInfo,		doom },			// HereticShareware
		{ retail,		&HereticGameInfo,		doom },			// HereticExtended
		{ retail,		&HereticGameInfo,		doom },			// Heretic
		{ shareware,	&SharewareGameInfo,		doom },			// DoomShareware
		{ retail,		&RetailGameInfo,		doom },			// UltimateDoom
		{ registered,	&RegisteredGameInfo,	doom }			// DoomRegistered
	};

	D_AddFile (iwadpath);

	if ((unsigned)type < NUM_IWAD_TYPES)
	{
		gamemode = Datas[type].Mode;
		gameinfo = *Datas[type].Info;
		gamemission = Datas[type].Mission;
		if (type == IWAD_HereticExtended)
		{
			gameinfo.flags |= GI_MENUHACK_EXTENDED;
		}
	}
	else
	{
		gamemode = undetermined;
	}
}

//==========================================================================
//
// ScanIWAD
//
// Scan the contents of an IWAD to determine which one it is
//==========================================================================

static EIWADType ScanIWAD (const char *iwad)
{
	static const char checklumps[][8] =
	{
		"E1M1",
		"E4M1",
		"MAP01",
		"TITLE",
		"REDTNT2",
		"CAMO1",
		{ 'E','X','T','E','N','D','E','D'},
		"E2M1","E2M2","E2M3","E2M4","E2M5","E2M6","E2M7","E2M8","E2M9",
		"E3M1","E3M2","E3M3","E3M4","E3M5","E3M6","E3M7","E3M8","E3M9",
		"DPHOOF","BFGGA0","HEADA1","CYBRA1",
		{ 'S','P','I','D','A','1','D','1' }
	};
#define NUM_CHECKLUMPS (sizeof(checklumps)/8)
	enum
	{
		Check_e1m1,
		Check_e4m1,
		Check_map01,
		Check_title,
		Check_redtnt2,
		Check_cam01,
		Check_Extended,
		Check_e2m1
	};
	int lumpsfound[NUM_CHECKLUMPS];
	size_t i;
	wadinfo_t header;
	FILE *f;

	memset (lumpsfound, 0, sizeof(lumpsfound));
	if ( (f = fopen (iwad, "rb")) )
	{
		fread (&header, sizeof(header), 1, f);
		if (header.Magic == IWAD_ID || header.Magic == PWAD_ID)
		{
			header.NumLumps = LONG(header.NumLumps);
			if (0 == fseek (f, LONG(header.InfoTableOfs), SEEK_SET))
			{
				for (i = 0; i < (size_t)header.NumLumps; i++)
				{
					wadlump_t lump;
					size_t j;

					if (0 == fread (&lump, sizeof(lump), 1, f))
						break;
					for (j = 0; j < NUM_CHECKLUMPS; j++)
						if (strnicmp (lump.Name, checklumps[j], 8) == 0)
							lumpsfound[j]++;
				}
			}
		}
		fclose (f);
	}

	if (lumpsfound[Check_map01])
	{
		if (lumpsfound[Check_redtnt2])
		{
			return IWAD_Doom2TNT;
		}
		else if (lumpsfound[Check_cam01])
		{
			return IWAD_Doom2Plutonia;
		}
		else
		{
			if (lumpsfound[Check_title])
			{
				return IWAD_Hexen;
			}
			else
			{
				return IWAD_Doom2;
			}
		}
	}
	else if (lumpsfound[Check_e1m1])
	{
		if (lumpsfound[Check_title])
		{
			if (!lumpsfound[Check_e2m1])
			{
				return IWAD_HereticShareware;
			}
			else
			{
				if (lumpsfound[Check_Extended])
				{
					return IWAD_HereticExtended;
				}
				else
				{
					return IWAD_Heretic;
				}
			}
		}
		else
		{
			for (i = Check_e2m1; i < NUM_CHECKLUMPS; i++)
			{
				if (!lumpsfound[i])
				{
					return IWAD_DoomShareware;
				}
			}
			if (i == NUM_CHECKLUMPS)
			{
				if (lumpsfound[Check_e4m1])
				{
					return IWAD_UltimateDoom;
				}
				else
				{
					return IWAD_DoomRegistered;
				}
			}
		}
	}
	return NUM_IWAD_TYPES;	// Don't know
}

//==========================================================================
//
// CheckIWAD
//
// Tries to find an IWAD from a set of know IWAD names, and checks the
// contents of each one found to determine which game it belongs to.
// Returns the number of new wads found in this pass (does not count wads
// found from a previous call).
// 
//==========================================================================

static int CheckIWAD (const char *doomwaddir, WadStuff *wads)
{
	const char *slash;
	char iwad[512];
	int i;
	int numfound;

	numfound = 0;

	slash = (doomwaddir[0] && doomwaddir[strlen (doomwaddir)-1] != '/') ? "/" : "";

	// Search for a pre-defined IWAD
	for (i = IWADNames[0] ? 0 : 1; IWADNames[i]; i++)
	{
		if (wads[i].Path == NULL)
		{
			sprintf (iwad, "%s%s%s", doomwaddir, slash, IWADNames[i]);
			FixPathSeperator (iwad);
			if (FileExists (iwad))
			{
				wads[i].Type = ScanIWAD (iwad);
				if (wads[i].Type != NUM_IWAD_TYPES)
				{
					wads[i].Path = copystring (iwad);
					numfound++;
				}
			}
		}
	}

	return numfound;
}

//==========================================================================
//
// CheckIWADinEnvDir
//
// Checks for an IWAD in a directory specified in an environment variable.
//
//==========================================================================

static int CheckIWADinEnvDir (const char *envname, WadStuff *wads)
{
	char dir[512];
	const char *envdir = getenv (envname);

	if (envdir)
	{
		strcpy (dir, envdir);
		FixPathSeperator (dir);
		if (dir[strlen(dir) - 1] != '/')
			strcat (dir, "/");
		return CheckIWAD (dir, wads);
	}
	return false;
}

//==========================================================================
//
// IdentifyVersion
//
// Tries to find an IWAD in one of four directories under DOS or Win32:
//	  1. Current directory
//	  2. Executable directory
//	  3. $DOOMWADDIR
//	  4. $HOME
//
// Under UNIX OSes, the search path is:
//	  1. Current directory
//	  2. $DOOMWADDIR
//	  3. $HOME/.zdoom
//	  4. The share directory defined at compile time (/usr/local/share/zdoom)
//
// The search path can be altered by editing the IWADSearch.Directories
// section of the config file.
//
//==========================================================================

// [NightFang] - added
extern	char		zd_iwadname[256];		//Kilgore: need space if the wad name contains path info

static EIWADType IdentifyVersion (void)
{
	WadStuff wads[sizeof(IWADNames)/sizeof(char *)];
	const char *iwadparm = Args.CheckValue ("-iwad");
	char *homepath = NULL;
	size_t numwads;
	int pickwad;
	size_t i;
	bool iwadparmfound = false;

	memset (wads, 0, sizeof(wads));

	if (iwadparm)
	{
		char *custwad = new char[strlen (iwadparm) + 5];
		strcpy (custwad, iwadparm);
		FixPathSeperator (custwad);
		if (CheckIWAD (custwad, wads))
		{ // -iwad parameter was a directory
			delete[] custwad;
			iwadparm = NULL;
		}
		else
		{
			DefaultExtension (custwad, ".wad");
			iwadparm = custwad;
			IWADNames[0] = iwadparm;
			CheckIWAD ("", wads);
		}
	}

	if (iwadparm == NULL || wads[0].Path == NULL)
	{
		if (GameConfig->SetSection ("IWADSearch.Directories"))
		{
			const char *key;
			const char *value;

			while (GameConfig->NextInSection (key, value))
			{
				if (stricmp (key, "Path") == 0)
				{
					if (*value == '$')
					{
						if (stricmp (value + 1, "progdir") == 0)
						{
							CheckIWAD (progdir, wads);
						}
						else
						{
							CheckIWADinEnvDir (value + 1, wads);
						}
					}
#ifdef unix
					else if (*value == '~' && (*(value + 1) == 0 || *(value + 1) == '/'))
					{
						homepath = GetUserFile (*(value + 1) ? value + 2 : value + 1);
						CheckIWAD (homepath, wads);
						delete[] homepath;
						homepath = NULL;
					}
#endif
					else
					{
						CheckIWAD (value, wads);
					}
				}
			}
		}
	}

	if (iwadparm != NULL && wads[0].Path != NULL)
	{
		iwadparmfound = true;
	}

	for (i = numwads = 0; i < sizeof(IWADNames)/sizeof(char *); i++)
	{
		if (wads[i].Path != NULL)
		{
			if (i != numwads)
				wads[numwads] = wads[i];
			numwads++;
		}
	}

	if (numwads == 0)
	{
		I_FatalError ("Cannot find a game IWAD (doom.wad, doom2.wad, heretic.wad, etc.).\n"
					  "Did you install ZServ properly?");
	}

	pickwad = 0;

	// [NightFang] - Dont display that stupid box! EVER AGAIN!
	//if (!iwadparmfound && numwads > 1 && queryiwad)
	//{
	//	pickwad = I_PickIWad (wads, numwads);
	//}

	if (pickwad < 0)
		exit (0);

	SetIWAD (wads[pickwad].Path, wads[pickwad].Type);

	// [NightFang] - setup the iwad
	strcpy(zd_iwadname, wads[pickwad].Path);

	for (i = 0; i < numwads; i++)
	{
		delete[] wads[i].Path;
	}

	if (homepath)
		delete[] homepath;

	return wads[pickwad].Type;
}

//==========================================================================
//
// BaseFileSearch
//
// If a file does not exist at <file>, looks for it in the directories
// specified in the config file. Returns the path to the file, if found,
// or NULL if it could not be found.
//
//==========================================================================

static const char *BaseFileSearch (const char *file, const char *ext)
{
	static char wad[PATH_MAX];

	if (FileExists (file))
	{
		return file;
	}

	if (GameConfig->SetSection ("FileSearch.Directories"))
	{
		const char *key;
		const char *value;

		while (GameConfig->NextInSection (key, value))
		{
			if (stricmp (key, "Path") == 0)
			{
				const char *dir;
				char *homepath = NULL;

				if (*value == '$')
				{
					if (stricmp (value + 1, "progdir") == 0)
					{
						dir = progdir;
					}
					else
					{
						dir = getenv (value + 1);
					}
				}
#ifdef unix
				else if (*value == '~' && (*(value + 1) == 0 || *(value + 1) == '/'))
				{
					homepath = GetUserFile (*(value + 1) ? value + 2 : value + 1);
					dir = homepath;
				}
#endif
				else
				{
					dir = value;
				}
				if (dir != NULL)
				{
					sprintf (wad, "%s%s%s", dir, dir[strlen (dir) - 1] != '/' ? "/" : "", file);
					if (homepath != NULL)
					{
						delete[] homepath;
						homepath = NULL;
					}
					if (FileExists (wad))
					{
						return wad;
					}
				}
			}
		}
	}

	// Retry, this time with a default extension
	if (ext != NULL)
	{
		static char tmp[PATH_MAX];
		strcpy (tmp, file);
		DefaultExtension (tmp, ext);
		return BaseFileSearch (tmp, NULL);
	}
	return NULL;
}

//==========================================================================
//
// ConsiderPatches
//
// Tries to add any deh/bex patches from the command line.
// Kilgore: this stuff should not happen in client/server mode
//
//==========================================================================
//
//bool ConsiderPatches (const char *arg, const char *ext)
//{
//	bool noDef = false;
//	DArgs *files = Args.GatherFiles (arg, ext, false);
//
//	if (files->NumArgs() > 0)
//	{
//		int i;
//		const char *f;
//
//		for (i = 0; i < files->NumArgs(); ++i)
//		{
//			if ( (f = BaseFileSearch (files->GetArg (i), ".deh")) )
//				DoDehPatch (f, false);
//			else if ( (f = BaseFileSearch (files->GetArg (i), ".bex")) )
//				DoDehPatch (f, false);
//		}
//		noDef = true;
//	}
//	delete files;
//	return noDef;
//}

// [Dash|RD] -- void SV_CHROOT (void)
//		
//		chroots the ZDaemon process into it's own jail and changes its current user.
//		This is purely for security.
//
//		Variables:	Name		Type		Default		Desc.
//		------------------------------------------------------------------------------------------------
//				sv_chroot	Bool		False		Run zserv in a chroot?
//				sv_chrootjail	String		./zserv_jail	Folder to chroot to
//				sv_chrootuid	Int		65534		UID of user to switch to after
//										a successful chroot. Defaults
//										to user `nobody'.

#ifdef unix

void SV_CHROOT (void)
{
     if (!sv_chroot)  // chroot isn't selected.
	return;

     if (getuid() != 0)
             Printf("WARNING: Not attempting a chroot; process not run as root.\n");
     else
     {
        Printf("NOTICE: zserv has been chrooted for security reasons.\n");
        if (chroot(sv_chrootjail) == -1) 
           { 
		perror("zserv chroot"); 
		Printf("WARNING: chroot failed!\n"); 
		return; 
	   }
        chdir("/");
	// Change to another user. Even chroot will not save you from yourself as root.
	if (setuid(sv_chrootuid) == -1)
	   { 
	     perror("zserv setuid"); exit(-1);   // We must kill zserv -- allowing users further would give
						 // them a false sense of security.
	   }
     }
}

void SV_SETPLATFORM_ID()
{
	char    *pjnk,*pend;
        int             c;
        FILE    *f;

        strcpy(Platform_ID,DETAILED_STRVERSION);
        if ( (f=popen("uname -sr","r")) != NULL )
        {
           strcat(Platform_ID," on ");
           pjnk = Platform_ID + strlen(Platform_ID);
           pend = Platform_ID+sizeof(Platform_ID)-2;
           for (;;)
           {
              if ( (c=getc(f)) == EOF || c=='\n') break;
              if (pjnk>=pend) break;
              *pjnk++ = (char) c;
           }
           pclose(f);
           *pjnk = '\0';
        }

}

#endif

//==========================================================================
//
// D_MultiExec
//
//==========================================================================

void D_MultiExec (DArgs *list, bool usePullin)
{
	for (int i = 0; i < list->NumArgs(); ++i)
	{
		C_ExecFile (list->GetArg (i), usePullin);
	}
}

//==========================================================================
//
// D_DoomMain
//
//==========================================================================
extern int		do_stdin;

void D_DoomMain (void)
{
	int p, flags;
	char file[PATH_MAX];
	char *v;
	const char *wad;
	DArgs *execFiles;


#if defined(_MSC_VER) || defined(__GNUC__)
	TypeInfo::StaticInit ();
#endif
	
#ifdef _WIN32
	//Raider added this because zserv didn't know what the
	//program directory is, so it couldn't find the skins folder
	// Figure out what directory the program resides in.
	GetModuleFileName (NULL, progdir, 512);
	*(strrchr (progdir, '\\') + 1) = '\0';
	FixPathSeperator (progdir);
	//Printf("progdir=>%s<=\n", progdir);	
#endif

	// [nightfang]
	zd_countwad = 0;

	atterm (DObject::StaticShutdown);
	atterm (DoConsoleAtExit);

	gamestate = GS_STARTUP;

	SetLanguageIDs ();

	rngseed = (DWORD)time (NULL);
	M_FindResponseFile ();
	M_LoadDefaults ();			// load before initing other systems
	Z_Init ();					// [RH] Init zone heap after loading config file

	if (Args.CheckParm("-log"))		genlog_control("gen");	// General logging

	// Make sure zdaemon.wad is always loaded,
	wad = BaseFileSearch ("zdaemon.wad", NULL);
	if (wad)
		D_AddFile (wad);
	else
		I_FatalError ("Cannot find zdaemon.wad");

	I_SetTitleString (IWADTypeNames[IdentifyVersion ()]);
	GameConfig->DoGameSetup (GameNames[gameinfo.gametype]);

	// Run automatically executed files
	execFiles = new DArgs;
	GameConfig->AddAutoexec (execFiles, GameNames[gameinfo.gametype]);
	D_MultiExec (execFiles, true);
	delete execFiles;

	// Run .cfg files at the start of the command line.
	execFiles = Args.GatherFiles (NULL, ".cfg", false);
	D_MultiExec (execFiles, true);
	delete execFiles;

	C_ExecCmdLineParams ();		// [RH] do all +set commands on the command line

	// [RH] Add any .wad files in the skins directory
	// Raider changed this. load_skins now enables or disables skins:
	if (load_skins > 0)
	{	
		#ifdef unix
			sprintf (file, "%sskins", SHARE_DIR);
		#else
			sprintf (file, "%sskins", progdir);
		#endif
		D_AddDirectory (file);

		const char *home = getenv ("HOME");
		if (home)
		{
			sprintf (file, "%s%s.zdoom/skins", home,
				home[strlen(home)-1] == '/' ? "" : "/");
			D_AddDirectory (file);
		}
	}

	if (!(gameinfo.flags & GI_SHAREWARE))
	{
		// Add common (global) wads
		D_AddConfigWads ("Global.Autoload");

		// Add game-specific wads
		sprintf (file, "%s.Autoload", GameNames[gameinfo.gametype]);
		D_AddConfigWads (file);
	}

	DArgs *files = Args.GatherFiles ("-file", ".wad", true);
	if (files->NumArgs() > 0)
	{
		// the files gathered are wadfile/lump names
		for (int i = 0; i < files->NumArgs(); i++)
		{
			//zd_countwad = true;
			D_AddFile (files->GetArg (i));
			//Printf("ZD_ADDING: %s\n", files->GetArg(i));

			// [NightFang] - make sure this wad is valid really quick
			{
				char	*filename;
				char	name[64];
				FILE	*f;

				filename = copystring(files->GetArg(i));
				FixPathSeperator (filename);
				strcpy (name, filename);
				DefaultExtension (name, ".wad");

				f = fopen(name, "r");
				if (f)
				{ 
					ZD_AddWad(name);
					fclose(f);
					//Printf (" couldn't open %s\n",filename);
					//return;
				}
			}
		}
	}
	delete files;

	W_InitMultipleFiles (&wadfiles);

	// [RH] Initialize localizable strings.
	GStrings.LoadStrings (W_GetNumForName ("LANGUAGE"), STRING_TABLE_SIZE, false);
	GStrings.Compact ();

	//P_InitXlat ();

	// [RH] Moved these up here so that we can do most of our
	//		startup output in a fullscreen console.

	I_Init ();

	// Base systems have been inited; enable cvar callbacks
	FBaseCVar::EnableCallbacks ();

	// [RH] Now that all text strings are set up,
	// insert them into the level and cluster data.
	G_SetLevelStrings ();
	
	// [RH] Parse through all loaded mapinfo lumps
	G_ParseMapInfo ();

	// [RH] Parse any SNDINFO lumps
	S_ParseSndInfo ();

	FActorInfo::StaticInit ();
	FActorInfo::StaticGameSet ();

	DecalLibrary.Clear ();
	DecalLibrary.ReadAllDecals ();

	// [RH] Try adding .deh and .bex files on the command line.
	// If there are none, try adding any in the config file.

	if (gameinfo.gametype == GAME_Doom)
	{
//Kilgore: forget about deh and bex patches in client/server mode
//		if (!ConsiderPatches ("-deh", ".deh") &&
//			!ConsiderPatches ("-bex", ".bex") &&
//			GameConfig->SetSection ("Doom.DefaultDehacked"))
//		{
//			const char *key;
//			const char *value;
//
//			while (GameConfig->NextInSection (key, value))
//			{
//				if (stricmp (key, "Path") == 0 && FileExists (value))
//				{
//					Printf ("Applying patch %s\n", value);
//					DoDehPatch (value, true);
//				}
//			}
//		}
//
		DoDehPatch (NULL, true);	// See if there's a patch in a PWAD
	}

	FActorInfo::StaticSetActorNums ();

	// [RH] User-configurable startup strings. Because BOOM does.
	if (GStrings(STARTUP1)[0])	Printf ("%s\n", GStrings(STARTUP1));
	if (GStrings(STARTUP2)[0])	Printf ("%s\n", GStrings(STARTUP2));
	if (GStrings(STARTUP3)[0])	Printf ("%s\n", GStrings(STARTUP3));
	if (GStrings(STARTUP4)[0])	Printf ("%s\n", GStrings(STARTUP4));
	if (GStrings(STARTUP5)[0])	Printf ("%s\n", GStrings(STARTUP5));


	bglobal.spawn_tries = 0;

	int nbotv = 0;
	const char *nbotss = Args.CheckValue ("-nbots");
	if (nbotss)
	{
		nbotv = atoi(nbotss);
		if (nbotv<0) nbotv=0;
		else if (nbotv>MAXPLAYERS) nbotv = 0;
	}
	if (nbotv>0)
	{
		bglobal.getspawned = NULL;
		bglobal.wanted_botnum = nbotv;
	}
	else
	{
		bglobal.getspawned = Args.GatherFiles ("-bots", "", false);
		if (bglobal.getspawned->NumArgs() == 0)
		{
			delete bglobal.getspawned;
			bglobal.getspawned = NULL;
			bglobal.wanted_botnum = 0;
		}
		else
		{
			bglobal.wanted_botnum = bglobal.getspawned->NumArgs();
		}
	}

	srand((unsigned) time(NULL));

	flags = dmflags;
		
	if (Args.CheckParm ("-nomonsters"))		flags |= DF_NO_MONSTERS;
	if (Args.CheckParm ("-respawn"))		flags |= DF_MONSTERS_RESPAWN;
	if (Args.CheckParm ("-fast"))			flags |= DF_FAST_MONSTERS;

	devparm = Args.CheckParm ("-devparm");

	if (Args.CheckParm ("-altdeath"))
	{
		deathmatch = 1;
		flags |= DF_ITEMS_RESPAWN;
	}
	else if (Args.CheckParm ("-deathmatch"))
	{
		deathmatch = 1;
		flags |= DF_WEAPONS_STAY | DF_ITEMS_RESPAWN;
	}

	dmflags = flags;

	// get skill / episode / map from parms
	strcpy (startmap, (gameinfo.flags & GI_MAPxx) ? "MAP01" : "E1M1");
	autostart = false;
				
	const char *val = Args.CheckValue ("-skill");
	if (val)
	{
		gameskill = val[0] - '1';
		autostart = true;
	}

	p = Args.CheckParm ("-warp");
	if (p && p < Args.NumArgs() - (1+(gameinfo.flags & GI_MAPxx ? 0 : 1)))
	{
		int ep, map;

		if (gameinfo.flags & GI_MAPxx)
		{
			ep = 1;
			map = atoi (Args.GetArg(p+1));
		}
		else 
		{
			ep = Args.GetArg(p+1)[0]-'0';
			map = Args.GetArg(p+2)[0]-'0';
			if ((unsigned)map > 9)
			{
				map = ep;
				ep = 1;
			}
		}

		strncpy (startmap, CalcMapName (ep, map), 8);
		autostart = true;
	}

	// [RH] Hack to handle +map
	p = Args.CheckParm ("+map");
	if (p && p < Args.NumArgs()-1)
	{
		if (W_CheckNumForName (Args.GetArg (p+1)) == -1)
		{
			Printf ("Can't find map %s\n", Args.GetArg (p+1));
		}
		else
		{
			strncpy (startmap, Args.GetArg (p+1), 8);
			Args.GetArg (p)[0] = '-';
			autostart = true;
		}
	}
	if (devparm)
		Printf (GStrings(D_DEVSTR));

#ifndef unix
	if (Args.CheckParm("-cdrom"))
	{
		Printf (GStrings(D_CDROM));
		mkdir ("c:\\zdoomdat", 0);
	}
#endif

	// turbo option  // [RH] (now a cvar)
	{
		UCVarValue value;

		value.String = Args.CheckValue ("-turbo");
		if (value.String == NULL)
			value.String = "100";
		else
			Printf ("turbo scale: %s%%\n", value.String);

		turbo.SetGenericRepDefault (value, CVAR_String);
	}

	v = Args.CheckValue ("-timer");
	if (v)
	{
		int timev = atoi(v);
		if (timev>0)
		{
			Printf ("Levels will end after %d minute%s.\n", timev, (timev > 1) ? "s" : "");
			timelimit = timev;
		}
	}

	if (Args.CheckParm("-avg"))
	{
		Printf ("Austin Virtual Gaming: Levels will end after 20 minutes\n");
		timelimit = 20;
	}

	if (Args.CheckParm("-noinput"))	do_stdin = 0;				// Don't read stdin
	if (Args.CheckParm("-dlog"))	dbglog_control("dbg");		// Debug logging
	if (Args.CheckParm("-clog"))	connlog_control("conn");	// Connection logging
	if (Args.CheckParm("-flog"))	fraglog_control("frag");	// Frag logging
	if (Args.CheckParm("-wlog"))	weaponlog_control("weap");	// Frag/weapon logging

	Printf ("Init DOOM refresh subsystem.\n");
	R_Init ();

	Printf ("Init Playloop state.\n");
	P_Init ();

	Printf ("Setting up sound.\n");
	S_Init ();

	Printf ("Checking network game status.\n");
	D_CheckNetGame ();

	// [NightFang] - Start the network api and create our sockets
	Printf("ZD_StartNetwork: starting network and creating sockets\n");
	ZD_StartNetwork();

	// [RH] Lock any cvars that should be locked now that we're
	// about to begin the game.
	FBaseCVar::EnableNoSet ();

	// [RH] Run any saved commands from the command line or autoexec.cfg now.
	gamestate = GS_FULLCONSOLE;
	Net_NewMakeTic ();
	DObject::BeginFrame ();
	DThinker::RunThinkers ();
	DObject::EndFrame ();
	gamestate = GS_STARTUP;

	// [NightFang] - check CFG validation
	ZD_ValidateCFG();
	parameter_checks();

	// Initialize the client buffers *after* we know maxclients; in this
	// way we save some memory and we can use pretty high values for the
	// NUMPLAYERS contant.
	InitNetCommon2();

		// [NightFang] - modded up a bit
		G_NewInit ();
			
		// [NightFang] - start the first map in our sequence if we're using one
		if (zd_cycleactive)
			G_InitNew(zd_mapcycle[zd_nextmap]);
		else
			G_InitNew (startmap);

		// [NightFang] - have the next map ready
		ZD_NextMap();

	connlog_printf("zserv startup");

	if (Args.CheckParm ("-debugfile"))
	{
		char	filename[20];
		sprintf (filename,"debug%i.txt",consoleplayer);
		Printf ("debug output to: %s\n",filename);
		debugfile = fopen (filename,"w");
	}

	atterm (D_QuitNetGame);		// killough

	// [NightFang] - Extra params that must be set in order for ZDaemon to
	// ingore single player type of play
	multiplayer = true;

	// [NightFang] - let the master server know we're here
	ZD_UpdateMaster();

	SV_ReadLocalBans();
	SV_FetchMasterBanList();

#ifdef unix
	// [Dash|RD] -- Grab *nix version information before chrooting.
	SV_SETPLATFORM_ID();
	// [Dash|RD] -- chroot zserv for security purposes. (Only works on UNIX platforms)
	SV_CHROOT();
#endif

        Printf("Starting watchdog thread\n");
        atterm (watchdog_stop);
        watchdog_start();
        set_alarm_to_next_midnight();

#ifndef	USEGUI
	// [NightFang] - done elsewhere
	D_DoomLoop ();		// never returns
#endif
}

//==========================================================================
//
// STAT fps
//
// Displays statistics about rendering times
//
//==========================================================================

ADD_STAT (fps, out)
{
	sprintf (out,
		"frame=%04.1f ms  walls=%04.1f ms  planes=%04.1f ms  masked=%04.1f ms",
		(double)FrameCycles * SecondsPerCycle * 1000,
		(double)WallCycles * SecondsPerCycle * 1000,
		(double)PlaneCycles * SecondsPerCycle * 1000,
		(double)MaskedCycles * SecondsPerCycle * 1000
		);
}

//==========================================================================
//
// STAT wallcycles
//
// Displays the minimum number of cycles spent drawing walls
//
//==========================================================================

static cycle_t bestwallcycles = INT_MAX;

ADD_STAT (wallcycles, out)
{
	if (WallCycles && WallCycles < bestwallcycles)
		bestwallcycles = WallCycles;
	sprintf (out, "%lu", bestwallcycles);
}

//==========================================================================
//
// CCMD clearwallcycles
//
// Resets the count of minimum wall drawing cycles
//
//==========================================================================

CCMD (clearwallcycles)
{
	bestwallcycles = INT_MAX;
}

#if 0
// To use these, also uncomment the clock/unclock in wallscan
static cycle_t bestscancycles = INT_MAX;

ADD_STAT (scancycles, out)
{
	if (WallScanCycles && WallScanCycles < bestscancycles)
		bestscancycles = WallScanCycles;
	sprintf (out, "%d", bestscancycles);
}

CCMD (clearscancycles)
{
	bestscancycles = INT_MAX;
}
#endif
