//  <- declarations <-
/*
 * This code implements the MD5 message-digest algorithm
 * invented by Ron Rivest. This code was written by Colin
 * Plumb and updated by Ian Jackson. C++ wrapper and 80x86
 * assembly by Hippocrates Sendoukas (Kilgore). No copyright
 * is claimed. This code is in the public domain; do with it
 * what you wish.
 */
#include "md5.h"
#include <string.h>             /* for memcpy() */

typedef unsigned UWORD32;

#define md5byte unsigned char

//  -> declarations ->
//  <- byteSwap <-

#if BIG_ENDIAN

static void
byteSwap(UWORD32 *buf, unsigned words)
{
	md5byte *p = (md5byte *)buf;

	do
	{
		*buf++ = (UWORD32)((unsigned)p[3] << 8 | p[2]) << 16 |
							((unsigned)p[1] << 8 | p[0]);
		p += 4;
	}
	while (--words);
}

#else

#define byteSwap(buf,words)

#endif

//  -> byteSwap ->
//  <- MD5Init <-

/*
 * Start MD5 accumulation.  Set bit count to 0 and buffer to mysterious
 * initialization constants.
 */
static void
MD5Init(struct MD5Context *ctx)
{
	ctx->buf[0] = 0x67452301;
	ctx->buf[1] = 0xefcdab89;
	ctx->buf[2] = 0x98badcfe;
	ctx->buf[3] = 0x10325476;
	ctx->bytes[0] = 0;
	ctx->bytes[1] = 0;
}

//  -> MD5Init ->
//  <- MD5Transform <-

#if _WIN32

/* The four core MD5 functions */

#define MD5_F1(w,x,y,z,inval,c,s)	\
			__asm	mov	eax,z		\
			__asm	add	w,inval		\
			__asm	xor	eax,y		\
			__asm	and	eax,x		\
			__asm	add	w,c			\
			__asm	xor	eax,z		\
			__asm	add	w,eax		\
			__asm	rol	w,s			\
			__asm	add	w,x			\

#define MD5_F2(w,x,y,z,inval,c,s)	\
			__asm	mov	eax,y		\
			__asm	add	w,inval		\
			__asm	xor	eax,x		\
			__asm	and	eax,z		\
			__asm	add	w,c			\
			__asm	xor	eax,y		\
			__asm	add	w,eax		\
			__asm	rol	w,s			\
			__asm	add	w,x

#define MD5_F3(w,x,y,z,inval,c,s)	\
			__asm	mov	eax,x		\
			__asm	add	w,inval		\
			__asm	xor	eax,y		\
			__asm	add	w,c			\
			__asm	xor	eax,z		\
			__asm	add	w,eax		\
			__asm	rol	w,s			\
			__asm	add	w,x

#define MD5_F4(w,x,y,z,inval,c,s)	\
			__asm	mov	eax,z		\
			__asm	add	w,inval		\
			__asm	not	eax			\
			__asm	or	eax,x		\
			__asm	add	w,c			\
			__asm	xor	eax,y		\
			__asm	add	w,eax		\
			__asm	rol	w,s			\
			__asm	add	w,x

#define VA	edi
#define VB	ebx
#define VC	ecx
#define VD	edx
#define VIN(i)	[esi+4*i]

/*
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
static void
MD5Transform(UWORD32 buf[4], UWORD32 const vin[16])
{
	__asm	mov	esi,buf
	__asm	mov	VA,[esi]
	__asm	mov	VB,[esi+1*4]
	__asm	mov	VC,[esi+2*4]
	__asm	mov	VD,[esi+3*4]
	__asm	mov	esi,vin

	MD5_F1( VA, VB, VC, VD, VIN( 0), 0d76aa478h,  7)
	MD5_F1( VD, VA, VB, VC, VIN( 1), 0e8c7b756h, 12)
	MD5_F1( VC, VD, VA, VB, VIN( 2), 0242070dbh, 17)
	MD5_F1( VB, VC, VD, VA, VIN( 3), 0c1bdceeeh, 22)
	MD5_F1( VA, VB, VC, VD, VIN( 4), 0f57c0fafh,  7)
	MD5_F1( VD, VA, VB, VC, VIN( 5), 04787c62ah, 12)
	MD5_F1( VC, VD, VA, VB, VIN( 6), 0a8304613h, 17)
	MD5_F1( VB, VC, VD, VA, VIN( 7), 0fd469501h, 22)
	MD5_F1( VA, VB, VC, VD, VIN( 8), 0698098d8h,  7)
	MD5_F1( VD, VA, VB, VC, VIN( 9), 08b44f7afh, 12)
	MD5_F1( VC, VD, VA, VB, VIN(10), 0ffff5bb1h, 17)
	MD5_F1( VB, VC, VD, VA, VIN(11), 0895cd7beh, 22)
	MD5_F1( VA, VB, VC, VD, VIN(12), 06b901122h,  7)
	MD5_F1( VD, VA, VB, VC, VIN(13), 0fd987193h, 12)
	MD5_F1( VC, VD, VA, VB, VIN(14), 0a679438eh, 17)
	MD5_F1( VB, VC, VD, VA, VIN(15), 049b40821h, 22)

	MD5_F2( VA, VB, VC, VD, VIN( 1), 0f61e2562h,  5)
	MD5_F2( VD, VA, VB, VC, VIN( 6), 0c040b340h,  9)
	MD5_F2( VC, VD, VA, VB, VIN(11), 0265e5a51h, 14)
	MD5_F2( VB, VC, VD, VA, VIN( 0), 0e9b6c7aah, 20)
	MD5_F2( VA, VB, VC, VD, VIN( 5), 0d62f105dh,  5)
	MD5_F2( VD, VA, VB, VC, VIN(10), 002441453h,  9)
	MD5_F2( VC, VD, VA, VB, VIN(15), 0d8a1e681h, 14)
	MD5_F2( VB, VC, VD, VA, VIN( 4), 0e7d3fbc8h, 20)
	MD5_F2( VA, VB, VC, VD, VIN( 9), 021e1cde6h,  5)
	MD5_F2( VD, VA, VB, VC, VIN(14), 0c33707d6h,  9)
	MD5_F2( VC, VD, VA, VB, VIN( 3), 0f4d50d87h, 14)
	MD5_F2( VB, VC, VD, VA, VIN( 8), 0455a14edh, 20)
	MD5_F2( VA, VB, VC, VD, VIN(13), 0a9e3e905h,  5)
	MD5_F2( VD, VA, VB, VC, VIN( 2), 0fcefa3f8h,  9)
	MD5_F2( VC, VD, VA, VB, VIN( 7), 0676f02d9h, 14)
	MD5_F2( VB, VC, VD, VA, VIN(12), 08d2a4c8ah, 20)

	MD5_F3( VA, VB, VC, VD, VIN( 5), 0fffa3942h,  4)
	MD5_F3( VD, VA, VB, VC, VIN( 8), 08771f681h, 11)
	MD5_F3( VC, VD, VA, VB, VIN(11), 06d9d6122h, 16)
	MD5_F3( VB, VC, VD, VA, VIN(14), 0fde5380ch, 23)
	MD5_F3( VA, VB, VC, VD, VIN( 1), 0a4beea44h,  4)
	MD5_F3( VD, VA, VB, VC, VIN( 4), 04bdecfa9h, 11)
	MD5_F3( VC, VD, VA, VB, VIN( 7), 0f6bb4b60h, 16)
	MD5_F3( VB, VC, VD, VA, VIN(10), 0bebfbc70h, 23)
	MD5_F3( VA, VB, VC, VD, VIN(13), 0289b7ec6h,  4)
	MD5_F3( VD, VA, VB, VC, VIN( 0), 0eaa127fah, 11)
	MD5_F3( VC, VD, VA, VB, VIN( 3), 0d4ef3085h, 16)
	MD5_F3( VB, VC, VD, VA, VIN( 6), 004881d05h, 23)
	MD5_F3( VA, VB, VC, VD, VIN( 9), 0d9d4d039h,  4)
	MD5_F3( VD, VA, VB, VC, VIN(12), 0e6db99e5h, 11)
	MD5_F3( VC, VD, VA, VB, VIN(15), 01fa27cf8h, 16)
	MD5_F3( VB, VC, VD, VA, VIN( 2), 0c4ac5665h, 23)

	MD5_F4( VA, VB, VC, VD, VIN( 0), 0f4292244h,  6)
	MD5_F4( VD, VA, VB, VC, VIN( 7), 0432aff97h, 10)
	MD5_F4( VC, VD, VA, VB, VIN(14), 0ab9423a7h, 15)
	MD5_F4( VB, VC, VD, VA, VIN( 5), 0fc93a039h, 21)
	MD5_F4( VA, VB, VC, VD, VIN(12), 0655b59c3h,  6)
	MD5_F4( VD, VA, VB, VC, VIN( 3), 08f0ccc92h, 10)
	MD5_F4( VC, VD, VA, VB, VIN(10), 0ffeff47dh, 15)
	MD5_F4( VB, VC, VD, VA, VIN( 1), 085845dd1h, 21)
	MD5_F4( VA, VB, VC, VD, VIN( 8), 06fa87e4fh,  6)
	MD5_F4( VD, VA, VB, VC, VIN(15), 0fe2ce6e0h, 10)
	MD5_F4( VC, VD, VA, VB, VIN( 6), 0a3014314h, 15)
	MD5_F4( VB, VC, VD, VA, VIN(13), 04e0811a1h, 21)
	MD5_F4( VA, VB, VC, VD, VIN( 4), 0f7537e82h,  6)
	MD5_F4( VD, VA, VB, VC, VIN(11), 0bd3af235h, 10)
	MD5_F4( VC, VD, VA, VB, VIN( 2), 02ad7d2bbh, 15)
	MD5_F4( VB, VC, VD, VA, VIN( 9), 0eb86d391h, 21)

	__asm	mov	esi,buf
	__asm	add	[esi],VA
	__asm	add	[esi+1*4],VB
	__asm	add	[esi+2*4],VC
	__asm	add	[esi+3*4],VD
}

#else

/* The four core functions - F1 is optimized somewhat */

/* #define F1(x, y, z) (x & y | ~x & z) */
#define F1(x, y, z) (z ^ (x & (y ^ z)))
#define F2(x, y, z) F1(z, x, y)
#define F3(x, y, z) (x ^ y ^ z)
#define F4(x, y, z) (y ^ (x | ~z))

/* This is the central step in the MD5 algorithm. */
#define MD5STEP(f,w,x,y,z,in,s) \
	 (w += f(x,y,z) + in, w = (w<<s | w>>(32-s)) + x)

/*
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
static void
MD5Transform(UWORD32 buf[4], UWORD32 const in[16])
{
	register UWORD32 a, b, c, d;

	a = buf[0];
	b = buf[1];
	c = buf[2];
	d = buf[3];

	MD5STEP(F1, a, b, c, d, in[0] + 0xd76aa478, 7);
	MD5STEP(F1, d, a, b, c, in[1] + 0xe8c7b756, 12);
	MD5STEP(F1, c, d, a, b, in[2] + 0x242070db, 17);
	MD5STEP(F1, b, c, d, a, in[3] + 0xc1bdceee, 22);
	MD5STEP(F1, a, b, c, d, in[4] + 0xf57c0faf, 7);
	MD5STEP(F1, d, a, b, c, in[5] + 0x4787c62a, 12);
	MD5STEP(F1, c, d, a, b, in[6] + 0xa8304613, 17);
	MD5STEP(F1, b, c, d, a, in[7] + 0xfd469501, 22);
	MD5STEP(F1, a, b, c, d, in[8] + 0x698098d8, 7);
	MD5STEP(F1, d, a, b, c, in[9] + 0x8b44f7af, 12);
	MD5STEP(F1, c, d, a, b, in[10] + 0xffff5bb1, 17);
	MD5STEP(F1, b, c, d, a, in[11] + 0x895cd7be, 22);
	MD5STEP(F1, a, b, c, d, in[12] + 0x6b901122, 7);
	MD5STEP(F1, d, a, b, c, in[13] + 0xfd987193, 12);
	MD5STEP(F1, c, d, a, b, in[14] + 0xa679438e, 17);
	MD5STEP(F1, b, c, d, a, in[15] + 0x49b40821, 22);

	MD5STEP(F2, a, b, c, d, in[1] + 0xf61e2562, 5);
	MD5STEP(F2, d, a, b, c, in[6] + 0xc040b340, 9);
	MD5STEP(F2, c, d, a, b, in[11] + 0x265e5a51, 14);
	MD5STEP(F2, b, c, d, a, in[0] + 0xe9b6c7aa, 20);
	MD5STEP(F2, a, b, c, d, in[5] + 0xd62f105d, 5);
	MD5STEP(F2, d, a, b, c, in[10] + 0x02441453, 9);
	MD5STEP(F2, c, d, a, b, in[15] + 0xd8a1e681, 14);
	MD5STEP(F2, b, c, d, a, in[4] + 0xe7d3fbc8, 20);
	MD5STEP(F2, a, b, c, d, in[9] + 0x21e1cde6, 5);
	MD5STEP(F2, d, a, b, c, in[14] + 0xc33707d6, 9);
	MD5STEP(F2, c, d, a, b, in[3] + 0xf4d50d87, 14);
	MD5STEP(F2, b, c, d, a, in[8] + 0x455a14ed, 20);
	MD5STEP(F2, a, b, c, d, in[13] + 0xa9e3e905, 5);
	MD5STEP(F2, d, a, b, c, in[2] + 0xfcefa3f8, 9);
	MD5STEP(F2, c, d, a, b, in[7] + 0x676f02d9, 14);
	MD5STEP(F2, b, c, d, a, in[12] + 0x8d2a4c8a, 20);

	MD5STEP(F3, a, b, c, d, in[5] + 0xfffa3942, 4);
	MD5STEP(F3, d, a, b, c, in[8] + 0x8771f681, 11);
	MD5STEP(F3, c, d, a, b, in[11] + 0x6d9d6122, 16);
	MD5STEP(F3, b, c, d, a, in[14] + 0xfde5380c, 23);
	MD5STEP(F3, a, b, c, d, in[1] + 0xa4beea44, 4);
	MD5STEP(F3, d, a, b, c, in[4] + 0x4bdecfa9, 11);
	MD5STEP(F3, c, d, a, b, in[7] + 0xf6bb4b60, 16);
	MD5STEP(F3, b, c, d, a, in[10] + 0xbebfbc70, 23);
	MD5STEP(F3, a, b, c, d, in[13] + 0x289b7ec6, 4);
	MD5STEP(F3, d, a, b, c, in[0] + 0xeaa127fa, 11);
	MD5STEP(F3, c, d, a, b, in[3] + 0xd4ef3085, 16);
	MD5STEP(F3, b, c, d, a, in[6] + 0x04881d05, 23);
	MD5STEP(F3, a, b, c, d, in[9] + 0xd9d4d039, 4);
	MD5STEP(F3, d, a, b, c, in[12] + 0xe6db99e5, 11);
	MD5STEP(F3, c, d, a, b, in[15] + 0x1fa27cf8, 16);
	MD5STEP(F3, b, c, d, a, in[2] + 0xc4ac5665, 23);

	MD5STEP(F4, a, b, c, d, in[0] + 0xf4292244, 6);
	MD5STEP(F4, d, a, b, c, in[7] + 0x432aff97, 10);
	MD5STEP(F4, c, d, a, b, in[14] + 0xab9423a7, 15);
	MD5STEP(F4, b, c, d, a, in[5] + 0xfc93a039, 21);
	MD5STEP(F4, a, b, c, d, in[12] + 0x655b59c3, 6);
	MD5STEP(F4, d, a, b, c, in[3] + 0x8f0ccc92, 10);
	MD5STEP(F4, c, d, a, b, in[10] + 0xffeff47d, 15);
	MD5STEP(F4, b, c, d, a, in[1] + 0x85845dd1, 21);
	MD5STEP(F4, a, b, c, d, in[8] + 0x6fa87e4f, 6);
	MD5STEP(F4, d, a, b, c, in[15] + 0xfe2ce6e0, 10);
	MD5STEP(F4, c, d, a, b, in[6] + 0xa3014314, 15);
	MD5STEP(F4, b, c, d, a, in[13] + 0x4e0811a1, 21);
	MD5STEP(F4, a, b, c, d, in[4] + 0xf7537e82, 6);
	MD5STEP(F4, d, a, b, c, in[11] + 0xbd3af235, 10);
	MD5STEP(F4, c, d, a, b, in[2] + 0x2ad7d2bb, 15);
	MD5STEP(F4, b, c, d, a, in[9] + 0xeb86d391, 21);

	buf[0] += a;
	buf[1] += b;
	buf[2] += c;
	buf[3] += d;
}


#endif

//  -> MD5Transform ->
//  <- MD5Update <-

/*
 * Update context to reflect the concatenation of another buffer full
 * of bytes.
 */
static void
MD5Update(struct MD5Context *ctx, md5byte const *buf, unsigned len)
{
	UWORD32 t;

	/* Update byte count */

	t = ctx->bytes[0];
	if ((ctx->bytes[0] = t + len) < t)
		ctx->bytes[1]++;        /* Carry from low to high */

	t = 64 - (t & 0x3f);    /* Space available in ctx->in (at least 1) */
	if (t > len)
	{
		memcpy((md5byte *)ctx->in + 64 - t, buf, len);
		return;
	}
	/* First chunk is an odd size */
	memcpy((md5byte *)ctx->in + 64 - t, buf, t);
	byteSwap(ctx->in, 16);
	MD5Transform(ctx->buf, ctx->in);
	buf += t;
	len -= t;

	/* Process data in 64-byte chunks */
	while (len >= 64)
	{
		memcpy(ctx->in, buf, 64);
		byteSwap(ctx->in, 16);
		MD5Transform(ctx->buf, ctx->in);
		buf += 64;
		len -= 64;
	}

	/* Handle any remaining bytes of data. */
	memcpy(ctx->in, buf, len);
}

//  -> MD5Update ->
//  <- MD5Final <-

/*
 * Final wrapup - pad to 64-byte boundary with the bit pattern
 * 1 0* (64-bit count of bits processed, MSB-first)
 */
static void
MD5Final(md5byte digest[16], struct MD5Context *ctx)
{
	int count = ctx->bytes[0] & 0x3f;       /* Number of bytes in ctx->in */
	md5byte *p = (md5byte *)ctx->in + count;

	/* Set the first char of padding to 0x80.  There is always room. */
	*p++ = 0x80;

	/* Bytes of padding needed to make 56 bytes (-8..55) */
	count = 56 - 1 - count;

	if (count < 0)		/* Padding forces an extra block */
	{
		memset(p, 0, count + 8);
		byteSwap(ctx->in, 16);
		MD5Transform(ctx->buf, ctx->in);
		p = (md5byte *)ctx->in;
		count = 56;
	}
	memset(p, 0, count);
	byteSwap(ctx->in, 14);

	/* Append length in bits and transform */
	ctx->in[14] = ctx->bytes[0] << 3;
	ctx->in[15] = ctx->bytes[1] << 3 | ctx->bytes[0] >> 29;
	MD5Transform(ctx->buf, ctx->in);

	byteSwap(ctx->buf, 4);
	memcpy(digest, ctx->buf, 16);
	memset(ctx, 0, sizeof(ctx));    /* In case it's sensitive */
}

//  -> MD5Final ->
//  <- MD5::MD5 <-

MD5::MD5()
{
	init();
}

//  -> MD5::MD5 ->
//  <- MD5::init <-

void
MD5::init()
{
	MD5Init(&mdc);
}

//  -> MD5::init ->
//  <- MD5::update <-

void
MD5::update(unsigned char const *buf, unsigned len)
{
	MD5Update(&mdc,buf,len);
}

//  -> MD5::update ->
//  <- MD5::final <-

void
MD5::final(unsigned char *digest)
{
	MD5Final(digest,&mdc);
}

//  -> MD5::final ->
