#include <stdio.h>

#ifdef	WIN32
#include <winsock2.h>
#include <ws2tcpip.h>
#endif

#include "doomdef.h"
#include "doomstat.h"
#include "gstrings.h"
#include "d_player.h"
#include "c_cvars.h"
#include "c_dispatch.h"
#include "c_bind.h"
#include "c_console.h"
#include "g_game.h"
#include "g_level.h"
#include "gi.h"
#include "p_local.h"
#include "i_system.h"
#include "g_game.h"
#include "gstrings.h"
#include "s_sound.h"
#include "m_argv.h"
#include "cmdlib.h"
#include "version.h"
#include "m_misc.h"
#include "w_wad.h"
#include "sv_main.h"
#include "log.h"
#include <time.h>


wadpath_t			wadpaths[NUM_WAD_PATHS];
int					numusedwads=0;				// Pwads only
char				gamename[64];
char				zd_iwadname[512];			//Kilgore: need space if the wad name contains path info

static const char	master_addr[] = "zdaemon.ath.cx";

// Huffman for old server compat
void HuffEncode(unsigned char *in,unsigned char *out,int inlen,int *outlen);
//static unsigned char huffbuff[65536];
int		outlen;

CVAR (String, hostname, "ZDaemon 1.07 Server", CVAR_SERVERINFO|CVAR_ARCHIVE)
CVAR (String, website, "http://www.zdaemon.org/", CVAR_SERVERINFO|CVAR_ARCHIVE)
CVAR (String, email, "raider@zdaemon.org", CVAR_SERVERINFO|CVAR_ARCHIVE)
CVAR (String, optional_wads, "", CVAR_SERVERINFO|CVAR_ARCHIVE)

EXTERN_CVAR (Int,	master_advertise)
EXTERN_CVAR (Int,	force_password)		//Bond
EXTERN_CVAR (Int,	timelimit)			//Bond
EXTERN_CVAR (Int,	fraglimit)			//Bond
EXTERN_CVAR (Int,	teamscorelimit)		//Kilgore
EXTERN_CVAR (Int,	maxteams)			//Kilgore
EXTERN_CVAR (Float,	teamdamage)			//Kilgore
EXTERN_CVAR (Int,	minplayers)			//Kilgore
EXTERN_CVAR (Int,	removebotswhenhumans)	//Kilgore
EXTERN_CVAR (Float, sv_gravity)
EXTERN_CVAR (Float, sv_aircontrol)

const netadr_t * const masteraddr(void)
{
	static bool initialized = false;
	static netadr_t		loc_masteraddr;

	if (!initialized)
	{
		NET_StringToAdr(master_addr, &loc_masteraddr);
		I_SetPort(&loc_masteraddr, MASTER_PORT);
		initialized = true;
	}
	return &loc_masteraddr;
}

const netadr_t * const localhost_addr(void)
{
	static bool initialized = false;
	static netadr_t		taddr;

	if (!initialized)
	{
		NET_StringToAdr("127.0.0.1", &taddr);
		I_SetPort(&taddr, localport);
		initialized = true;
	}
	return &taddr;
}


#define LAUNCHER_EXTENDED_INFO	0x01020304		//Bond: signal to launcher that more info is available

CCMD(wads)
{
	int			i;
	for (i=0; i<numusedwads; i++)
	{
		Printf("%d. %s\n", i+1, wadpaths[i].name);
	}

	Printf("\nUsing %d external wads.\n", numusedwads);
}

//Kilgore: return pointer to filename excluding any path info
static char *plain_filename(const char *fname)
{
	char *p,*q;

	if ( (p=strrchr((char *)fname,'/')) != NULL ) p++;
	else p = (char *)fname;
	if ( (q=strrchr(p,'\\')) != NULL ) q++;
	else q = p;
	if ( (p=strrchr(q,':')) != NULL ) p++;
	else p = q;
	return p;
}

//Kilgore: safe string copying. destsz is the number
//of chars (not necessarily bytes).
static void strcpymax(char *dest,char *src,size_t destsz)
{
	size_t	n = strlen(src);
	if (n>=destsz) n = destsz-1;
	memcpy(dest,src,n*sizeof(dest[0]));
	dest[n] = '\0';
}

//Kilgore: cleaned up the code and made it safer
void ZD_AddWad(char	*wadfile)
{
	if (numusedwads>=NUM_WAD_PATHS) return;
	strcpymax( wadpaths[numusedwads].name, plain_filename(wadfile),
		       sizeof(wadpaths[0].name) / sizeof(wadpaths[0].name[0]) );
	numusedwads++;
}

void printErrorExit(char *text)
{
	// Close the network down
	ZD_CloseNetwork();
	I_FatalError(text);  
}

void GetGameName(void)
{
	strcpy(gamename,
				(gameinfo.gametype == GAME_Doom) ?
					(gamemode != commercial) ? "DOOM" : "DOOM II" :
				(gameinfo.gametype == GAME_Heretic) ? "Heretic" :
				(gameinfo.gametype == GAME_Hexen) ? "Hexen" :
				"ERROR!" );
}


//
// ZD_UpdateMaster
//

void ZD_UpdateMaster(void)
{

	// Respect the master_advertize setting 
	if (master_advertise == 0) return;

	// update master every 30 seconds
	if (gametic % (TICRATE*30))
		return;

	/*
	 * Kilgore: sometimes the unix server's internal clock goes haywire
	 * and starts ticking at an extremely fast rate. This results in a
	 * DOS (and even DDOS) to the master. At the same time, the server
	 * is unusable even though it has not crashed. The code below tries
	 * to detect this condition; if it recognizes it, then it simply
	 * shuts down the server. This is the best course of action as we
	 * don't want to destroy the master; besides, on most unix setups,
	 * the server will be automatically restarted as soon as it goes
	 * down for any reason.
	 */
	{
		FILE			*f;
		unsigned long		cur_time;
		struct tm		*lt;
		static unsigned		clock_too_fast = 0;
		static unsigned long	last_time = 0;

		cur_time = time(NULL);
		if (cur_time - last_time < 20)
		{
			if (++clock_too_fast >= 2)
			{
				if ( (f=fopen("zserv.crashlog","a")) != NULL )
				{
					lt = localtime( (time_t *) &cur_time);
					fprintf(f,"%04d-%02d-%02d %02d:%02d:%02d : Clock went haywire\n",
						lt->tm_year+1900, lt->tm_mon+1, lt->tm_mday,
						lt->tm_hour, lt->tm_min, lt->tm_sec);
					fclose(f);
				}
				//
				// Make it crash on purpose; it should be restarted automatically
				//
				lt = NULL;
				if (lt->tm_year)
					Printf("UNREACHABLE\n");
			}
		}
		else
		{
			clock_too_fast = 0;
		}
		last_time = cur_time;
	}

	ZDOP.Init();
	ZDOP.WriteLong(MASTER_CHALLENGE);
	ZDOP.WriteByte(VERSION);
	ZDOP.WriteWord(localport);	//Kilgore: support for NAPT
	ZDOP.SendTo(masteraddr(),false);
}


void ZD_SendInfo(void)
{
	int			i, numclients, gametype;
	UCVarValue	charttl;
	char		servername[128], filelocation[128], eemail[32];
	static char jnk[256];
#ifdef unix
	extern char Platform_ID[255]; 	// [Dash|RD] From d_main.cpp 
	EXTERN_CVAR (Bool, sv_chroot);	// ...   
#endif

	// Server name
	charttl = hostname.GetGenericRep (CVAR_String);
	// [NightFang] - Send the uncensored (good idea?) hostname
	strcpymax(servername, charttl.String, sizeof(servername)/sizeof(servername[0]));

	// Website URL
	charttl = website.GetGenericRep (CVAR_String);
	strcpymax(filelocation, charttl.String, sizeof(filelocation)/sizeof(filelocation[0]));

	// Email
	charttl = email.GetGenericRep (CVAR_String);
	strcpymax(eemail, charttl.String,sizeof(eemail)/sizeof(eemail[0]));

	// # of players connected
	numclients = ZD_ClientCount();

	// Also get the games name
	GetGameName();

	ZDOP.Init();
	ZDOP.WriteLong(MASTER_CHALLENGE);
	ZDOP.WriteString(servername);
	ZDOP.WriteByte(numclients);
	ZDOP.WriteByte(maxclients);
	ZDOP.WriteString(level.mapname);
	ZDOP.WriteByte(numusedwads);
	for (i=0; i<numusedwads; i++)
		ZDOP.WriteString(wadpaths[i].name);

	gametype =	
	#if __CTF__
				(ctf) ?			3 : // CTF
	#endif
				(teamplay) ?	1 :	// Team DM
				(deathmatch) ?	0 :	// Deathmatch (FFA)
								2 ;	// Coop. games

	ZDOP.WriteByte(gametype);
	ZDOP.WriteString(gamename);
	ZDOP.WriteString(plain_filename(zd_iwadname));	// Kilgore
	
	ZDOP.WriteByte((byte)gameskill);
	ZDOP.WriteString(filelocation);
	ZDOP.WriteString(eemail);
	ZDOP.WriteLong(dmflags);
	ZDOP.WriteLong(dmflags2);
	ZDOP.WriteByte(numclients);
			
	for (i=0; i<=top_client; i++)
	{
		if (!playeringame[i])	continue;
		ZDOP.WriteString(players[i].userinfo.netname);
		ZDOP.WriteShort((deathmatch) ? players[i].fragcount : players[i].killcount );
		ZDOP.WriteShort((players[i].isbot) ? 0 : clients[i].ping );
		ZDOP.WriteByte(players[i].level);
		ZDOP.WriteShort(players[i].minutesingame);
	}

	// [NF] - Now send the version num
	//==bond======================================================================
	ZDOP.WriteShort(VERSION);						//bond: the version number cannot be always 1.0

	ZDOP.WriteLong(LAUNCHER_EXTENDED_INFO);			// Signal about extended info.

	ZDOP.WriteByte((int)force_password);			// password status
	ZDOP.WriteByte(ZD_PlayerCount());				// current # of players in game 
	ZDOP.WriteByte(maxplayers);						// max # of players who can join

	int timeleft = timelimit - level.time/(TICRATE*60);
	if (timeleft<0) timeleft=0;

	ZDOP.WriteShort((int)timelimit);
	ZDOP.WriteShort(timeleft);
	ZDOP.WriteShort((int)fraglimit);

	for (i=0; i<=top_client; i++)
	{
		if (!playeringame[i]) continue;
		ZDOP.WriteByte(players[i].isbot);
		ZDOP.WriteByte(players[i].spectator);
		ZDOP.WriteByte(players[i].userinfo.team);
	}

	//==bond======================================================================

	// Kilgore: Send teamplay-related info

	ZDOP.WriteByte((int)maxteams);					// Max. *playing* teams
	ZDOP.WriteShort((int)teamscorelimit);			// Max. team score
	UCVarValue			td;
	td = teamdamage.GetGenericRep(CVAR_Float);
	ZDOP.WriteFloat(td.Float);						// Teamdamage value

	for (i=0; i<NUM_TEAMS; i++)
		ZDOP.WriteShort(TeamInfo[i].score);

	for (i=0; i<numusedwads; i++)
		ZDOP.WriteByte(WC_is_optional(wadpaths[i].name));

	ZDOP.WriteLong(LAUNCHER_EXTENDED_INFO);	// Signal about extended info.

	if (jnk[0]=='\0')
	{
		#ifdef unix
			// [Dash|RD] -- Now that we figure out this information when the server starts, this section can become a lot simpler.
			strcpy(jnk, Platform_ID);
			if (sv_chroot)
				strcat(jnk, " [Jailed]");
		#else
			OSVERSIONINFO vi;
	
			memset(&vi,0,sizeof(vi));
			vi.dwOSVersionInfoSize  = sizeof(vi);
			GetVersionEx(&vi);
			if (vi.dwPlatformId==VER_PLATFORM_WIN32_WINDOWS)
			{
				sprintf(jnk,"%s on Windows %s %u.%u.%u",
						DETAILED_STRVERSION,
						(vi.dwMinorVersion==0)	?	"95 " :
						(vi.dwMinorVersion==10)	?	"98 " :
													"ME" ,
						vi.dwMajorVersion , vi.dwMinorVersion, (unsigned) LOWORD(vi.dwBuildNumber) ); 
			}
			else
			{
				wsprintf(jnk,"%s on Windows %s %u.%u.%u",
						DETAILED_STRVERSION,
						 (vi.dwMajorVersion<=4) ?							"NT" :
						 (vi.dwMajorVersion==5 && vi.dwMinorVersion==0) ?	"2000 " :
						 (vi.dwMajorVersion==5 && vi.dwMinorVersion==1) ?	"XP" :
						 (vi.dwMajorVersion==5 && vi.dwMinorVersion==2) ?	"Server 2003 " :
																			"",
						 vi.dwMajorVersion , vi.dwMinorVersion, vi.dwBuildNumber );
			}
		#endif
		if (strlen(jnk)>63) jnk[64] = '\0';
	}

	ZDOP.WriteString(jnk);				// More detailed info about the
										// server build and O/S

	ZDOP.WriteLong(LAUNCHER_EXTENDED_INFO);	// Signal about extended info.
	UCVarValue			gr,ac;
	gr = sv_gravity.GetGenericRep(CVAR_Float);
	ZDOP.WriteFloat(gr.Float);
	ac = sv_aircontrol.GetGenericRep(CVAR_Float);
	ZDOP.WriteFloat(ac.Float);

	ZDOP.WriteLong(LAUNCHER_EXTENDED_INFO);	// Signal about extended info.
	for (i=0; i<=top_client; i++)
	{
		if (!playeringame[i]) continue;
		const char *c2 = (players[i].isbot) ? "_z" : clients[i].c2;
		if (!c2) c2 = "  ";
		ZDOP.WriteBytes(c2,2);
	}
	ZDOP.WriteByte((int)minplayers);
	ZDOP.WriteByte((int)removebotswhenhumans);

	ZDOP.WriteLong(0);						// For future extensions to the protocol
	ZDOP.SendTo(&net_from,false);
}


static void ZD_UpdatePlayerTime(player_t *p)
{
	unsigned long	t0,t1;

	if ( (t0=p->join_time) == 0 )
		t1 = 0;
	else
	{
		if ( (t1=time(NULL)) > t0)
			t1 -= t0;
		else
			t1 = 0;
	}
	t1 += p->past_time;
	if (t1)
		connlog_printf("TIME:\t%s\t%lu",p->userinfo.netname,t1);
	p->join_time = p->past_time = 0;
}

//
// ZD_UpdatePlayerStats - Sends the master server info about a player
// 
void ZD_UpdatePlayerStats(int pl)
{
	client_t		*cl;
	player_t		*p;

	p = players+pl;
	ZD_UpdatePlayerTime(p);
	if (p->experience==0) return;
	if (deathmatch && master_advertise!=0)
	{
		cl = clients + p->netid;
		// Create the stream (send the client address & name for verification) and send it
		ZDOP.Init();
		ZDOP.WriteLong(MASTER_PLAYERSTATS);
		ZDOP.WriteString(NET_AdrToString(&cl->address));
		ZDOP.WriteString(p->userinfo.netname);
		ZDOP.WriteShort(p->experience);
		ZDOP.SendTo(masteraddr(),false);
	}
	// Reset the experience
	p->experience = 0;
}


//
// ZD_UpdatePlayers - Does the above for ALL players
//

void ZD_UpdatePlayers(void)
{
	int		i;

	for (i=top_client; i>=0; i--)
	{
		if (!ZD_ValidClient(i)) continue;
		ZD_UpdatePlayerStats(i);
	}
}


#define MAX_INTERFACES	128

#if _WIN32

static int get_interfaces(struct in_addr *addr)
{
	SOCKET			s;
	int				i, n, nadded, localhost_added;
	DWORD			bytesReturned;
	INTERFACE_INFO	localAddr[MAX_INTERFACES];
	static unsigned char lh_addr[4] = { 127, 0, 0, 1 };

	localhost_added = 0;
	nadded = 0;
	if ((s = WSASocket (AF_INET, SOCK_DGRAM, IPPROTO_UDP, NULL, 0, 0)) != INVALID_SOCKET)
	{
		if (WSAIoctl(s, SIO_GET_INTERFACE_LIST, NULL, 0, &localAddr,
			sizeof (localAddr), &bytesReturned, NULL, NULL) == SOCKET_ERROR )
			bytesReturned = 0;
		closesocket(s);
		n = bytesReturned / sizeof(INTERFACE_INFO);
		for (i=0;  i<n;  i++)
		{
			if ( (localAddr[i].iiFlags & (IFF_UP | IFF_LOOPBACK)) == 0 ) continue;
			memcpy(addr+nadded, &(localAddr[i].iiAddress.AddressIn.sin_addr), sizeof(addr[0]) );
			if (addr[nadded].s_addr==0) continue;
			if (!memcmp(addr+nadded,lh_addr,4))
				localhost_added = 1;
			nadded++;
		}
	}
	if (!localhost_added && nadded<MAX_INTERFACES)
	{
		memcpy(addr+nadded,lh_addr,4);
		nadded++;
	}
	return nadded;
}


#else

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <string.h>

static int get_interfaces(struct in_addr *addr)
{    
	int						s, len, nadded, localhost_added;
	struct ifconf			ifc;
	struct ifreq			*ifr;
	struct sockaddr_in		*sinptr;
	char					*ptr, buf[8192];
	static unsigned char	lh_addr[4] = { 127, 0, 0, 1 };

	localhost_added = 0;
	nadded = 0;
	ifc.ifc_len = sizeof(buf);
	ifc.ifc_req = (struct ifreq *) buf;
	if ( (s=socket(AF_INET,SOCK_DGRAM,0))>=0  &&  ioctl(s,SIOCGIFCONF,&ifc)>=0)
	{
		for (ptr=buf;  ptr<buf+ifc.ifc_len;   )
		{
			ifr = (struct ifreq *) ptr;
			len = sizeof(struct sockaddr);
			#ifdef	HAVE_SOCKADDR_SA_LEN
				if (ifr->ifr_addr.sa_len > len)
					len = ifr->ifr_addr.sa_len;		/* length > 16 */
			#endif
			ptr += sizeof(ifr->ifr_name) + len;
			if (ifr->ifr_addr.sa_family==AF_INET  &&  nadded<MAX_INTERFACES)
			{
				sinptr = (struct sockaddr_in *) &ifr->ifr_addr;
				memcpy(addr+nadded, &sinptr->sin_addr, sizeof(addr[0]) );
				if (addr[nadded].s_addr)
				{
					if (!memcmp(addr+nadded,lh_addr,4))
						localhost_added = 1;
					nadded++;
				}
			}
		}
	}
	if (!localhost_added && nadded<MAX_INTERFACES)
	{
		memcpy(addr+nadded,lh_addr,4);
		nadded++;
	}
	return nadded;
}

#endif


bool
is_trusted_address(const netadr_t *a)
{
	int						i;
	static int				naddr = 0;
	static struct in_addr	addr[MAX_INTERFACES];

	if (!memcmp(a->ip,masteraddr()->ip,sizeof(a->ip)))
		return true;
	
	if (naddr==0)	naddr = get_interfaces(addr);
	for (i=naddr-1;  i>=0;  i--)
		if (!memcmp(a->ip,addr+i,sizeof(a->ip)))
			return true;
	return false;
}
