/*
** d_netinfo.cpp
** Manages transport of user and "server" cvars across a network
**
**---------------------------------------------------------------------------
** Copyright 1998-2001 Randy Heit
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
**
** 1. Redistributions of source code must retain the above copyright
**    notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote products
**    derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
** IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
** IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
** INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
** NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
** THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**---------------------------------------------------------------------------
**
*/

#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "doomtype.h"
#include "doomdef.h"
#include "doomstat.h"
#include "d_netinf.h"
#include "d_net.h"
#include "d_protocol.h"
#include "c_dispatch.h"
#include "v_palette.h"
#include "v_video.h"
#include "i_system.h"
#include "r_draw.h"
#include "r_state.h"
#include "gi.h"
#include "sv_main.h"

extern BOOL st_firsttime;

CVAR (Float,	autoaim,				5000.f,		CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (String,	name,					"Player",	CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (Color,	color,					0x40cf00,	CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (String,	skin,					"base",		CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (Int,		team,					255,		CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (String,	gender,					"male",		CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (Bool,		neverswitchonpickup,	false,		CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (Float,	movebob,				0.25f,		CVAR_USERINFO | CVAR_ARCHIVE);
CVAR (Float,	stillbob,				0.f,		CVAR_USERINFO | CVAR_ARCHIVE);

enum
{
	INFO_Name,
	INFO_Autoaim,
	INFO_Color,
	INFO_Skin,
	INFO_Team,
	INFO_Gender,
	INFO_NeverSwitchOnPickup,
	INFO_MoveBob,
	INFO_StillBob
};

//Kilgore
teaminfo_t	TeamInfo[NUM_TEAMS] =
{
	{ "Red",	0,	0xFF1010,	LS_LEAD_TIED	},
	{ "Blue",	0,	0x1010FF,	LS_LEAD_TIED	},
	{ "Green",	0,	0x00FF00,	LS_LEAD_TIED	},
	{ "Gold",	0,	0xFFDF40,	LS_LEAD_TIED	}
};

const char *GetTeamName(int i)		//Kilgore
{
	return (i>=0 && i<NUM_TEAMS) ? TeamInfo[i].name : "*ERROR*"; 
}                     
                     
void AssignTeamColor(userinfo_t *player)	//Kilgore
{
	int i;
	
	if (!teamplay) return;	
	i = player->team;
	player->color = (i>=0 && i<NUM_TEAMS) ? TeamInfo[i].color : 0xFFFFFF;
}


const char *GenderNames[3] = { "male", "female", "cyborg" };

static const char *UserInfoStrings[] =
{
	"name",
	"autoaim",
	"color",
	"skin",
	"team",
	"gender",
	"neverswitchonpickup",
	"movebob",
	"stillbob",
	NULL
};

int D_GenderToInt(const char *gender)
{
	return	(!stricmp(gender,"female")) ?	GENDER_FEMALE :
			(!stricmp(gender,"cyborg")) ?	GENDER_NEUTER :
											GENDER_MALE;
}


void D_ReadUserInfoStrings (int i, byte **stream, bool update)
{
	userinfo_t *info = &players[i].userinfo;
	char *ptr = *((char **)stream);
	char *breakpt;
	char *value;
	bool compact;
	int infotype = -1;

	if (*ptr++ != '\\')
		return;

	compact = (*ptr == '\\') ? ptr++, true : false;

	if (i>=0 && i<MAXPLAYERS)
	{
		for (;;)
		{
			breakpt = strchr (ptr, '\\');

			if (breakpt != NULL)
				*breakpt = 0;

			if (compact)
			{
				value = ptr;
				infotype++;
			}
			else
			{
				value = breakpt + 1;
				if ( (breakpt = strchr (value, '\\')) )
					*breakpt = 0;

				int j = 0;
				while (UserInfoStrings[j] && stricmp (UserInfoStrings[j], ptr) != 0)
					j++;
				if (UserInfoStrings[j] == NULL)
				{
					infotype = -1;
				}
				else
				{
					infotype = j;
				}
			}

			switch (infotype)
			{
			case INFO_Autoaim: {
				double angles;

				angles = atof (value);
				if (angles > 35.f || angles < 0.f)
				{
					info->aimdist = ANGLE_1*35;
				}
				else
				{
					info->aimdist = abs ((int)(angles * (float)ANGLE_1));
				}
								}
				break;

			case INFO_Name:
				{
					char oldname[MAXPLAYERNAME+1];

					strncpy (oldname, info->netname, MAXPLAYERNAME);
					oldname[MAXPLAYERNAME] = 0;
					strncpy (info->netname, value, MAXPLAYERNAME);
					info->netname[MAXPLAYERNAME] = 0;
					GUI_UpdateName(i);		// Kilgore - update server window
					if (update)
						Printf ("%s is now known as %s\n", oldname, info->netname);
				}
				break;

			case INFO_Team:
				info->team = atoi (value);
				if ((unsigned)info->team >= NUM_TEAMS)
					info->team = NUM_TEAMS-1;	//Kilgore: disallow loners & use last team as default
				if (update)
					Printf ("%s is now on Team %s\n", info->netname, GetTeamName(info->team));	//Kilgore
				break;

			case INFO_Color:
				info->color = V_GetColorFromString (NULL, value);
				R_BuildPlayerTranslation (i, info->color);
				// [Dash|RD] StatusBar dependancy removal.
				/*if (StatusBar != NULL && i == displayplayer)
				{
					StatusBar->AttachToPlayer (&players[i]);
				}*/
				break;

			case INFO_Skin:
				info->skin = R_FindSkin (value);
				if (players[i].mo != NULL)
				{
					if (players[i].mo->state->sprite.index ==
						GetDefaultByType (players[i].cls)->SpawnState->sprite.index)
					{ // Only change the sprite if the player is using a standard one
						players[i].mo->sprite = skins[info->skin].sprite;
					}
					players[i].skin = &skins[info->skin];
				}
				// [Dash|RD] StatusBar removal
				/*if (StatusBar != NULL && i == displayplayer)
				{
					StatusBar->SetFace (&skins[info->skin]);
				}*/
				break;

			case INFO_Gender:
				info->gender = D_GenderToInt (value);
				break;

			case INFO_NeverSwitchOnPickup:
				if (*value >= '0' && *value <= '9')
					info->neverswitch = atoi (value) ? true : false;
				else if (stricmp (value, "true") == 0)
					info->neverswitch = 1;
				else
					info->neverswitch = 0;
				break;

			case INFO_MoveBob:
				info->MoveBob = (fixed_t)(atof (value) * 65536.f);
				break;

			case INFO_StillBob:
				info->StillBob = (fixed_t)(atof (value) * 65536.f);
				break;

			default:
				break;
			}

			if (!compact)
			{
				*(value - 1) = '\\';
			}
			if (breakpt)
			{
				*breakpt = '\\';
				ptr = breakpt + 1;
			}
			else
			{
				break;
			}
		}
	}

	*stream += strlen (*((char **)stream)) + 1;
}

FArchive &operator<< (FArchive &arc, userinfo_t &info)
{
	if (arc.IsStoring ())
	{
		arc.Write (&info.netname, sizeof(info.netname));
		arc.Write (&info.team, sizeof(info.team));
	}
	else
	{
		arc.Read (&info.netname, sizeof(info.netname));
		arc.Read (&info.team, sizeof(info.team));
	}
	arc << info.aimdist << info.color << info.skin << info.gender << info.neverswitch;
	return arc;
}

CCMD (playerinfo)
{
	int i;

	if (argv.argc() < 2)
	{
		for (i=0;  i<=top_client;  i++)
		{
			if (playeringame[i])
				Printf ("%d. %s\n", i, players[i].userinfo.netname);
		}
	}
	else
	{
		i = atoi (argv[1]);
		if (i<0 || i>top_client) return;
		player_t *p = players + i;
		Printf ("Name:        %s\n", p->userinfo.netname);
		Printf ("Team:        %d\n", p->userinfo.team);
		Printf ("Aimdist:     %d\n", p->userinfo.aimdist);
		Printf ("Color:       %06x\n", p->userinfo.color);
		Printf ("Skin:        %d\n", p->userinfo.skin);
		Printf ("Gender:      %d\n", p->userinfo.gender);
		Printf ("NeverSwitch: %d\n", p->userinfo.neverswitch);
		Printf ("MoveBob:     %g\n", p->userinfo.MoveBob/65536.f);
		Printf ("StillBob:    %g\n", p->userinfo.StillBob/65536.f);
	}
}
