/*  <- data */
#include "ip2c.h"
#include <string.h>

#define VERSION		3		// Must be an integer

typedef struct
{
    unsigned long	ip1;
    unsigned long	ip2;
    char			c2[2];
} ip_s;

typedef struct
{
    char			c2[3];
    char			res2[4];
    const char		*cname;
} co_s;

static ip_s ip_list[] =
{
    #include "ip.dat"
    { 0xffffffffLU, 0xffffffffLU, { '~','~' } }
};

static co_s co_list[] =
{
    #include "co.dat"
    { "~~", "~~~", "~~~~~~" }
};

static const int N_IP = ( sizeof(ip_list) / sizeof(ip_list[0]) ) - 1;
static const int N_CO = ( sizeof(co_list) / sizeof(co_list[0]) ) - 1;

/*  -> data */
/*  <- ip2c_version		public */

int IP2C_EXPORT
ip2c_version(void)
{
    return VERSION;
}

/*  -> ip2c_version		public */
/*  <- find_country */

static co_s *
find_country(const char *c2)
{
    int left,right,mid,rc;

    left	= 0;
    right	= N_CO-1;
    while (left<=right)
    {
        mid = (left+right)/2;
        rc  = memcmp(c2,co_list[mid].c2,2);
        if	(rc==0)		return co_list+mid;
        else if	(rc>0)		left  = mid+1;
        else			right = mid-1;
    }
    return NULL;
}

/*  -> find_country */
/*  <- ip2c_info_c2		public		NEW */

int IP2C_EXPORT
ip2c_info_c2(const char *c2,const char **name,
             const char **flag, const char **flag2)
{
    co_s	*p;

    if (!c2) return 0;
    if ( (p=find_country(c2)) == NULL )	return 0;
    if (name)	*name = p->cname;
    if (flag)	*flag =  p->c2;
    if (flag2)	*flag2 = p->res2;
    return 1;
}

/*  -> ip2c_info_c2		public		NEW */
/*  <- ip_idx */

static int
ip_idx(unsigned long tgt)
{
    int left,right,mid;
    unsigned long v;

    //
    // First do a simple binary search using the beginning of
    // each IP range. Maybe we're lucky and the target value
    // is at the beginning of an IP range.
    //
    left	= 0;
    right	= N_IP-1;
    while (left<=right)
    {
        mid = (left+right)/2;
        v = ip_list[mid].ip1;
        if	(tgt==v)	return mid;
        else if	(tgt<v)		right = mid-1;
        else			left = mid+1;
    }

    //
    // At this point, the search has failed (ie., the target value does
    // not start an IP range). However, at this point, the "left" index
    // points right after the location where the target value should be
    // (if it happened to start an IP range). So, we backtrack a bit until
    // we find the first (highest) IP range that starts before the target
    // value. Then we can simply check if the target value lies in that
    // range. The loop below should normally execute only once...
    //
    do
    {
        left--;
        if (left<0) return -1;
    }
    while (ip_list[left].ip1>tgt);
    return (ip_list[left].ip2>=tgt) ? left : -1;
}

/*  -> ip_idx */
/*  <- string_to_ulong */

static unsigned long
string_to_ulong(const char *p)
{
    unsigned long lv;
    unsigned sv,i;

    lv = 0;
    for (i=0; i<4; i++)
    {
        if (*p<'0' || *p>'9') return 0;
        sv = 0;
        do
        {
            sv = 10*sv + (unsigned)(*p - '0');
            p++;
        }
        while (*p>='0' && *p<='9');
        if (sv>255) return 0;
        if (i<3)
        {
            if (*p!='.') return 0;
            p++;
        }
        lv <<= 8;
        lv += sv;
    }
    return lv;
}

/*  -> string_to_ulong */
/*  <- ip2c_info_n2		public		NEW */

int IP2C_EXPORT
ip2c_info_n2(unsigned long ip_address, const char **c2,
            const char **name, const char **flag, const char **flag2)
{
    int		i;
    co_s	*p;

    if ( (i=ip_idx(ip_address)) < 0 )			return 0;
    if ( (p=find_country(ip_list[i].c2)) == NULL )	return 0;
    if (name)	*name = p->cname;
    if (flag)	*flag = p->c2;
    if (flag2)	*flag2 = p->res2;
    if (c2)	*c2 = p->c2;
    return 1;
}

/*  -> ip2c_info_n2		public		NEW */
/*  <- ip2c_info_s2		public		NEW */

int IP2C_EXPORT
ip2c_info_s2(const char *ip_address, const char **c2,
            const char **name, const char **flag, const char **flag2)
{
    unsigned long v;

    v = string_to_ulong(ip_address);
    return (v) ? ip2c_info_n2(v,c2,name,flag,flag2) : 0;
}

/*  -> ip2c_info_s2		public		NEW */
/*  <- ip2c_info_n			public */

int IP2C_EXPORT
ip2c_info_n(unsigned long ip_address, const char **c2,
            const char **name, const char **flag)
{
    return ip2c_info_n2(ip_address,c2,name,flag,NULL);
}

/*  -> ip2c_info_n			public */
/*  <- ip2c_info_s			public */

int IP2C_EXPORT
ip2c_info_s(const char *ip_address, const char **c2,
            const char **name, const char **flag)
{
    return ip2c_info_s2(ip_address,c2,name,flag,NULL);
}

/*  -> ip2c_info_s			public */
#if __IP2C_DLL__
/*  <- DllEntryPoint */

BOOL WINAPI _CRT_INIT(HINSTANCE,DWORD,LPVOID);

BOOL WINAPI
DllEntryPoint(HINSTANCE hinst,DWORD reason,LPVOID p)
{
    #ifndef __TURBOC__
        if (reason==DLL_PROCESS_ATTACH || reason==DLL_THREAD_ATTACH)
            if(!_CRT_INIT(hinst,reason,p)) return FALSE;
        if (reason==DLL_PROCESS_DETACH || reason==DLL_THREAD_DETACH)
            if (!_CRT_INIT(hinst,reason,p)) return FALSE;
    #endif
    return TRUE;
}

/*  -> DllEntryPoint */
#endif
