#include "doomtype.h"
#include <stdio.h>
#include <stdarg.h>
#include "log.h"
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include "submit.h"
#ifdef unix
	#include <syslog.h>
#endif

/********************************************************
 * Common logging functions
 ********************************************************/

class log_c
{
	private:
	FILE		*f;
	bool		active, syslogging, notify_transitions;
	int			submission_id;
	char		log_basename[PATH_MAX+1];
	char		log_name[PATH_MAX+1];
	char		log_banner[64];

	public:
				log_c(const char *s,bool do_notify, int sub_id);
				~log_c();
	void		add(const char *msg,bool use_timestamp,bool use_newline);
	bool		open(const char *logname);
	void		close(void);
	bool		is_active(void)			{ return active; }
	bool		is_syslog(void)			{ return syslogging; }
	const char	*get_logname(void)		{ return log_name; }
	void		control(const char *logname);
	void		alarm(void);

	static unsigned	secs_to_midnight(void);
};

log_c::log_c(const char *s, bool do_notify, int sub_id)
{
	f = NULL;
	active = syslogging = false;
	memset(log_basename,0,sizeof(log_basename));
	memset(log_name,0,sizeof(log_name));
	strcpy(log_banner,s);
	notify_transitions = do_notify;
	submission_id = sub_id;
}

log_c::~log_c()
{
	close();
}

void log_c::close(void)
{
	char jnk[80];

	if (notify_transitions)
	{
		sprintf(jnk,"%s Log Stopped", log_banner);
		add(jnk,true,true);
	}
	if (f)
	{
		fclose(f);
		f = NULL;
	}
	active = syslogging = false;
}

bool log_c::open(const char *fname)
{
	time_t		t;
	struct tm	*lt;
	char		jnk[80];

	close();

	if (!fname || *fname=='\0') return false;

	if (fname!=log_basename) strcpy(log_basename,fname);
	#ifdef unix
		if (!strcmp(fname,"syslog"))
		{
			active = syslogging = true;
			strcpy(log_name,"System log");
			if (notify_transitions)
			{
				sprintf(jnk,"%s Log Started", log_banner);
				add(jnk,true,true);
			}
			return true;
		}
	#endif

	t = time(NULL);
	lt = localtime(&t);
	sprintf(log_name,"%s-%04d%02d%02d.log",fname,
			lt->tm_year+1900, lt->tm_mon+1, lt->tm_mday);
	if ( (f=fopen(log_name,"a")) != NULL )
	{
		fprintf(f,"\n");
		active = true;
		if (notify_transitions)
		{
			sprintf(jnk,"%s Log Started", log_banner);
			add(jnk,true,true);
		}
	}
	return active;
}

void log_c::add(const char *msg,bool use_timestamp,bool use_newline)
{
	time_t		t;
	struct tm	*lt;
	const char	*lf;
    
    if (!active) return;

	#ifdef unix
		if (syslogging)
		{
			syslog(LOG_INFO,msg);
			return;
		}
	#endif

	if (!f) return;

	lf = (use_newline) ? "\n" : "";

	if (use_timestamp)
	{
		t = time(NULL);
		lt = localtime(&t);
		fprintf(f,"%4d/%02d/%02d %02d:%02d:%02d %s%s",
				lt->tm_year+1900, lt->tm_mon+1, lt->tm_mday,
				lt->tm_hour, lt->tm_min, lt->tm_sec, msg, lf );
	}
	else
	{
		fprintf(f,"%s%s",msg, lf);
	}
	fflush(f);			//Kilgore: added this for robustness purposes
}

void log_c::control(const char *logname)
{
	if (is_active())
	{
		Printf (PRINT_HIGH,"%s logging off\n",log_banner);
		close();
	}
	else
	{
		if (open(logname))
			Printf(PRINT_HIGH, "%s logging started on %s\n",log_banner,get_logname());
		else
			Printf(PRINT_HIGH,"Cannot start %s log\n",log_banner);
	}
}

void log_c::alarm(void)
{
	char oldname[PATH_MAX+1];

	if (!is_active() || is_syslog())	return;
	strcpy(oldname,get_logname());
	control(NULL);
	control(log_basename);
	if (oldname[0] && strcmp(get_logname(),oldname))
	{
		if (submission_id>0)	submit_add(oldname,submission_id);
		else					dispose_log(oldname);
	}
}

unsigned log_c::secs_to_midnight(void)
{
	time_t		t;
	struct tm	*lt;
	unsigned	secs_from_midnight;

	t = time(NULL);
	lt = localtime(&t);

	// Compute how many seconds remain to midnight;
	// used for automatic log rotation.
	secs_from_midnight = lt->tm_sec + 60 * (lt->tm_min + 60*lt->tm_hour);
	return 86400 - secs_from_midnight;	//86400 = 24*60*60 = secs in a day
}

void set_alarm_to_next_midnight(void)
{
	unsigned t;

	t = log_c::secs_to_midnight();
	t += 10;				// Give it 10 more seconds (to make the date change
							// detection safer).
	t += (rand() % 110);	// Add some "salt" so that not everybody on the
							// same timezone hits the stats server at the same
							// time. Therefore, the next alarm should occur
							// sometime between 10 and 119 seconds after next 
							// midnight.
	set_alarm(t);
}

/********************************************************
 * Connection log
 ********************************************************/
      
static	log_c	connlog("Connection",true, 1);

void STACK_ARGS connlog_printf(const char *fmt,...)
{
	va_list	argptr;
	char	buf[2048];

	if (!connlog.is_active()) return;
	va_start(argptr,fmt);
	vsprintf(buf,fmt, argptr);
	va_end (argptr);
	connlog.add(buf,true,true);
}

void connlog_control(const char *fname)
{
	connlog.control(fname);
}

/********************************************************
 * Frag log
 ********************************************************/

static	log_c	fraglog("Frag",false, 0);

void STACK_ARGS fraglog_printf(const char *fmt,...)
{
	va_list	argptr;
	char	buf[512];

	if (!fraglog.is_active()) return;
	va_start(argptr,fmt);
	vsprintf(buf,fmt, argptr);
	va_end (argptr);
	fraglog.add(buf,false,true);
}

void fraglog_control(const char *fname)
{
	fraglog.control(fname);
}

/********************************************************
 * General log
 ********************************************************/

static	log_c	genlog("General",false, 0);

void STACK_ARGS genlog_printf(const char *fmt,...)
{
	va_list	argptr;
	char	buf[2048];

	if (!genlog.is_active()) return;
	va_start(argptr,fmt);
	vsprintf(buf,fmt, argptr);
	va_end (argptr);
	genlog.add(buf,false,false);
}

void genlog_control(const char *fname)
{
	genlog.control(fname);
}

/********************************************************
 * Frag/Weapon log
 ********************************************************/

static	log_c	weaponlog("Weapon",false, 2);

void STACK_ARGS weaponlog_printf(const char *fmt,...)
{
	va_list	argptr;
	char	buf[512];

	if (!weaponlog.is_active()) return;
	va_start(argptr,fmt);
	vsprintf(buf,fmt, argptr);
	va_end (argptr);
	weaponlog.add(buf,false,true);
}

void weaponlog_control(const char *fname)
{
	weaponlog.control(fname);
}

/********************************************************
 * Debug log
 ********************************************************/
      
static	log_c	dbglog("Debug",true, 0);

void STACK_ARGS dbglog_printf(const char *fmt,...)
{
	va_list	argptr;
	char	buf[2048];

	if (!dbglog.is_active()) return;
	va_start(argptr,fmt);
	vsprintf(buf,fmt, argptr);
	va_end (argptr);
	dbglog.add(buf,true,true);
}

void dbglog_control(const char *fname)
{
	dbglog.control(fname);
}

/********************************************************
 * Alarm and rotate all logs
 ********************************************************/
void log_alarm(void)
{
	genlog.alarm();
	fraglog.alarm();
	weaponlog.alarm();
	connlog.alarm();
	dbglog.alarm();
}

/********************************************************
 * Exit handler closing all logs
 ********************************************************/
void close_all_logs(void)
{
	connlog.close();
	fraglog.close();
	genlog.close();
	weaponlog.close();
	dbglog.close();
}


void enable_conn_weap_logs(void)
{
	if (!weaponlog.is_active())	weaponlog_control("weap");
	if (!connlog.is_active())	connlog_control("conn");
}
