#include "d_player.h"
#include "sv_main.h"
#include "c_dispatch.h"
#include "log.h"
#if _WIN32
	#include <process.h>
#else
	#include <sys/types.h>
	#include <unistd.h>
#endif
#include "inet.h"
#include <time.h>

/*
 * Ban / kicks
 */
#define		LOCAL_BANFILE			"zd_bans.txt"
#define		MASTER_BAN_URL			"http://www.zdaemon.org/bans/"

#define MAX_BANS	512				// That should be enough :-)

EXTERN_CVAR (Int,	master_advertise)

/*
 * Characteristics of a single ban. The IP address is a 4-byte value
 * as usual, with an important difference: for each IP address there
 * is an accompanying 4-byte "mask" array. If a mask byte is 1, then
 * the corresponding byte in the IP address is a wildard (ie., it will
 * match to any value supplied by the client for that particular byte.
 * This lets us use for example ip addresses of the form: 192.168.1.*
 * where the * character means that the ban should apply to any address
 * whose first three bytes are 192.168.1 and the last byte could be
 * anything. In the context of zserv, this can be helpful to admins,
 * since the IP address of a particular player will not remain constant
 * if the player is on a dial-up line.
 */
typedef struct
{
	byte	ip[4];			// ip address
	byte	mask[4];		// if 1 then corresponding byte in ip address is a wildcard
	bool	is_local;
	char	reason[1];		// Why we kick him
} ban_t;

class ban_cl
{
	private:

	ban_t	*bans[MAX_BANS];
	int		nbans;
	int		find(const byte *ip, const byte *mask);
	void	init();
	bool	address_matches(const byte *claddr,const byte *banip, const byte *mask);

	public:
			ban_cl();
			~ban_cl();
	void	flush();
	void	show();
	void	save(const char *fname);
	bool	add(const byte *ip,const byte *mask,const char *reason, bool is_local);
	bool	remove(const byte *ip,const byte *mask);
	void	enforce_to_all(void);
	bool	enforce_to_player(int cl);

	char	*ip_to_str(const byte *ip,const byte *mask);
	bool	str_to_ip(const char *s,byte *ip,byte *mask);
};

/*
 * Convert an ip address/mask to printable form
 */
char *ban_cl::ip_to_str(const byte *ip,const byte *mask)
{
	char		*p;
	int			i;
	static char buf[32];

	p = buf;
	for (i=0; i<4; i++)
	{
		if (i) *p++ = '.';
		if (mask[i])	p += sprintf(p,"*");
		else			p += sprintf(p,"%u",(unsigned)ip[i]);
	}
	return buf;
}

/*
 * Read an ip address/mask from a string. We allow the *
 * character in place of any byte the IP address. Also we
 * allow truncated addresses for compatibility purposes;
 * in those cases, the remaining bytes in the address will
 * be considered as wildcards.
 */
bool ban_cl::str_to_ip(const char *s,byte *ip, byte *mask)
{
	int			i;
	unsigned	v;
	char		c;

	memset(ip,0,4);
	memset(mask,1,4);

	for (i=0; i<4; i++)
	{
		c = *s++;
		if (c=='*')
		{
			mask[i] = 1;
			ip[i] = 0;
			c = *s++;
		}
		else
		{
			mask[i] = 0;
			if (c<'0' || c>'9') return false;
			v = 0;
			while (c>='0' && c<='9')
			{
				v = 10*v + (int)(c-'0');
				c = *s++;
			}
			if (v>255) return false;
			ip[i] = (byte) v;
		}
		if (i<3)
		{
			if (c=='\0')	// Truncated address: accept it for compatibility
			{				// with previous versions of zdaemon
				return true;
			}
			if (c!='.') return false;
		}
		else
		{
			if (c) return false;
		}
	}
	return true;
}

/*
 * Ban list constructor; simply set everything to zero
 */
ban_cl::ban_cl()
{
	init();
}

/*
 * Ban list destructor; clear all used memory and reset to zero
 */
ban_cl::~ban_cl()
{
	flush();
}

/*
 * Initialization stuff: reset all to zsero
 */
void ban_cl::init()
{
	nbans = 0;
	memset(bans,0,sizeof(bans));
}

/*
 * Clear all used memory and reset everything to zero
 */
void ban_cl::flush()
{
	int i;

	for (i=0; i<nbans; i++)
		free(bans[i]);
	init();
}

/*
 * Print the ban list on the screen
 */
void ban_cl::show(void)
{
	int		i;

	Printf("\n *********** BANS ***********\n");
	for (i=0; i<nbans; i++)
		Printf("%s (%s)\n", ip_to_str(bans[i]->ip,bans[i]->mask), bans[i]->reason);
	Printf(" ****************************\n");
}

/*
 * Save the ban list to a text file
 */
void ban_cl::save(const char *filename)
{
	FILE	*f;
	int		i;

	if ( (f=fopen(filename, "w")) == NULL )
	{
		Printf("*** Unable to create ban log file!\n");
		return;
	}
	fprintf(f, ";\n");
	fprintf(f, ";--------------------------\n");
	fprintf(f, "; ZDAEMON BAN FILE\n");
	fprintf(f, ";--------------------------\n");
	fprintf(f, ";\n;\n");

	for (i=0; i<nbans; i++)
		if (bans[i]->is_local)
			fprintf(f, "%s#%s\n", ip_to_str(bans[i]->ip,bans[i]->mask), bans[i]->reason);

	fclose(f);
}

/*
 * Find the ban that corresponds to the specified IP address and mask.
 * Returns -1 if not found.
 */
int ban_cl::find(const byte *ip, const byte *mask)
{
	int i;

	for (i=nbans-1; i>=0; i--)
	{
		if (!memcmp(bans[i]->ip,ip,4) && !memcmp(bans[i]->mask,mask,4))
			break;
	}
	return i;
}

/*
 * Add a ban to the ban linst and complain if something goes wrong
 */
bool ban_cl::add(const byte *ip, const byte *mask, const char *reason, bool is_local)
{
	const char	*r;
	ban_t		*pb;

	if ( find(ip,mask) >=0 )
	{
		Printf("Duplicate ban (%s)\n",ip_to_str(ip,mask));
		return false;
	}

	if (nbans >= MAX_BANS)
	{
		Printf("Too many bans\n");
		return false;
	}

	r = (!reason || *reason=='\0') ? " " : reason;

	if ( (pb = (ban_t *) malloc(sizeof(ban_t)+strlen(r))) == NULL )
	{
		Printf("Out of memory while adding bans\n");
		return false;
	}
	memcpy(pb->ip,ip,4);
	memcpy(pb->mask,mask,4);
	strcpy(pb->reason,r);
	pb->is_local = is_local;
	bans[nbans++] = pb;

	Printf("%s added to banlist\n", ip_to_str(ip,mask) );
	return true;
}

/*
 * Remove a ban from the ban list.
 */
bool ban_cl::remove(const byte *ip, const byte *mask)
{
	int		i;

	if ( (i=find(ip,mask)) < 0 ) return false;
	free(bans[i]);
	if (i<nbans-1)	bans[i] = bans[nbans-1];
	bans[nbans-1] = NULL;
	nbans--;
	return true;
}

/*
 * Determine if the client's address matches the IP address and mask of a single ban.
 */
bool ban_cl::address_matches(const byte *claddr,const byte *banip, const byte *mask)
{
	int i;

	for (i=0; i<4; i++)
	{
		if (mask[i]) continue;
		if (claddr[i]!=banip[i]) return false;
	}
	return true;
}

/*
 * Enforce all bans to a single player; if any ban matches, kick him out.
 * Returns true if a match was found, and false otherwise.
 */
bool ban_cl::enforce_to_player(int cl)
{
	int				i;
	byte			*client_ip;
	player_t		*p;
	char			*pname,string[1024];
	static byte		mask[4];		// the static attribute initializes it to zero

	if (!clients[cl].connected) return false;
	client_ip = clients[cl].address.ip;
	for (i=0; i<nbans; i++)
	{
		if (address_matches(client_ip,bans[i]->ip,bans[i]->mask))
		{
			sprintf(string, "BANNED: %s", bans[i]->reason);
			ZDOP.Init();
			ZDOP.WriteByte(sv_kick);
			ZDOP.WriteString(string);
			ZDOP.ToPlayer(cl);
			ZD_SendPacket(cl);
			p = players + cl;
			pname = (p->userinfo.netname[0]) ? p->userinfo.netname : ip_to_str(client_ip,mask);
			ZD_Printf("%s banned (%s)\n", pname, bans[i]->reason);
			connlog_printf("%s banned (%s)", pname, bans[i]->reason );
			ZD_DisconnectClient(cl);
			return true;
		}
	}
	return false;
}

/*
 * Enforce all bans to all players
 */
void ban_cl::enforce_to_all(void)
{
	int i;

	for (i=top_client;  i>=0;  i--)
	{
		if (!ZD_ValidClient(i)) continue;
		enforce_to_player(i);
	}
}

/*
 * The ban list: make it a static variable so that it's not
 * visible from outside this file, and also so that its constructor
 * and destructor are called automatically.
 */
static ban_cl ban_list;

static void
readbans_file(const char *fname)
{
	FILE	*f;
	int		nadded;
	byte	ip[4],mask[4];
	char	*reason, *pbuf, buf[1024];
	
	Printf("Reading bans from %s...",fname);
	if ( (f=fopen(fname, "r")) == NULL )
	{
		Printf("File not found\n");
		return;
	}
	Printf("\n");
	nadded = 0;
	while (fgets(buf, sizeof(buf)-1, f))
	{
		if ( (pbuf=strchr(buf,'\n')) != NULL )		// remove trailing LF
			*pbuf = '\0';

		if ( (pbuf=strchr(buf,'\r')) != NULL )		// remove trailing CR
			*pbuf = '\0';

		if ( (pbuf=strchr(buf,';')) != NULL )		// remove comments
			*pbuf = '\0';

		pbuf = buf;
		while (*pbuf==' ' || *pbuf=='\t')			// remove whitespace
			pbuf++;
		if (*pbuf == '\0')  continue;				// empty line

		reason = strchr(pbuf,'#');
		if (reason)	*reason++ = '\0';
		else reason = " ";

		if (!ban_list.str_to_ip(pbuf,ip,mask))
		{
			Printf("Bad IP address (%s)\n",pbuf);
			continue;
		}
		if (ban_list.add(ip,mask,reason,true))
			nadded++;
	}
	fclose(f);
	Printf("Added %d bans\n",nadded);
}

static void
readbans_mem(char *membuf)
{
	int		nadded;
	byte	ip[4],mask[4];
	char	*reason, *pbuf, *nextline;

	if (!membuf) return;
	Printf("Reading global bans...\n");
	nadded = 0;
	while (*membuf)
	{
		nextline = strchr(membuf,'\n');
		if (nextline) *nextline++ = '\0';
		else nextline = membuf + strlen(membuf);

		if ( (pbuf=strchr(membuf,'\r')) != NULL )	// remove trailing CR
			*pbuf = '\0';

		if ( (pbuf=strchr(membuf,';')) != NULL )	// remove comments
			*pbuf = '\0';

		pbuf = membuf;
		while (*pbuf==' ' || *pbuf=='\t')			// remove whitespace
			pbuf++;
		if (*pbuf)
		{
			reason = strchr(pbuf,'#');
			if (reason)	*reason++ = '\0';
			else reason = " ";
			if (!ban_list.str_to_ip(pbuf,ip,mask))
				Printf("Bad IP address (%s)\n",pbuf);
			else if (ban_list.add(ip,mask,reason,false))
				nadded++;
		}

		membuf = nextline;
	}
	Printf("Added %d bans\n",nadded);
}

/*
 * SV_ReadLocalBans - Reads & parses the local ban list into memory
 */
void SV_ReadLocalBans(void)
{
	readbans_file(LOCAL_BANFILE);
}

/*
 * Concatenate all command arguments into a single string
 */
static char *collect_args(FCommandLine &argv)
{
	int			i, last;
	char		*p;
	static char	buf[1024];

	memset(buf, 0, sizeof(buf));
	p = buf;
	last = argv.argc()-1;
	for (i=2; i<=last; i++)
	{
		if (i>2) *p++ = ' ';
		p += sprintf(p,"%s",argv[i]);
	}
	return buf;
}

/*
 * Kick out a player
 */
void SV_KickPlayer(int pnum, char *reason)
{
	player_t	*p;
	char		*s;
	
	p = players+pnum;
	s = p->userinfo.netname;

	// We cant kick bots! Duh
	if (p->isbot)
	{
		ZD_Printf("%s kicked. (%s)\n", s, reason);
		bglobal.ClearBot(pnum);
		return;
	}
	ZDOP.Init();
	ZDOP.WriteByte(sv_kick);
	ZDOP.WriteString(reason);
	ZDOP.ToPlayer(pnum);
	ZD_SendPacket(pnum);

	ZD_Printf("%s was kicked from the game (%s)\n", s, reason );
	connlog_printf("%s was kicked from the game (%s)", s, reason);

	ZD_DisconnectClient(pnum);
}

/*
 * COMMAND - kick Sends a simple message to the client to disconnect
 *
 * [Dash|RD] -- Rearranged to remove a goto.
 *
 */
CCMD(kick)
{
	int	pnum;

	if ((argv.argc() < 3) || (sscanf(argv[1],"%d",&pnum)!=1 || pnum < 0 || pnum>=MAXPLAYERS))
	{
		Printf (PRINT_HIGH, "\nKICK <player num> <reason>\n");
		Printf (PRINT_HIGH, "    Kicks a player off the server\n");
		return;
	} 
	if (!playeringame[pnum])
	{
		Printf(PRINT_HIGH, "\n ** Player %d not found!\n", pnum);
		return;
	}
	SV_KickPlayer(pnum, collect_args(argv));
}

/*
 * COMMAND - banlist Displays the current list of IP bans
 */
CCMD(banlist)
{
	ban_list.show();
}

/*
 * COMMAND - addban Adds an ip address to the ban list
 *
 * [Dash|RD] -- Rearranged to remove a goto.
 *
 */
CCMD(addban)
{
	byte ip[4],mask[4];

	if (argv.argc() < 3 || (!ban_list.str_to_ip(argv[1],ip,mask)))
	{
		Printf ("\nADDBAN <ip> <reason>\n");
		Printf ("\n     Bans an IP address from this server.\n\n");
		return;
	}
	if (ban_list.add(ip,mask,collect_args(argv),true))
	{
		ban_list.enforce_to_all();
		ban_list.save(LOCAL_BANFILE);
	}
}

/*
 * COMMAND - killban Removes an ip address from the ban list
 *
 * [Dash|RD] -- Rearranged to remove a goto.
 *
 */
CCMD(killban)
{
	byte ip[4],mask[4];

	if (argv.argc() < 2 || (!ban_list.str_to_ip(argv[1],ip,mask)))
	{
		Printf("\nKILLBAN <ip>\n");
		Printf("\n      Removes a banned IP from the shit list.\n\n");
		return;
	}
	if (ban_list.remove(ip,mask))
	{
		Printf("%s unbanned.\n", ban_list.ip_to_str(ip,mask));
		ban_list.save(LOCAL_BANFILE);
	}
	else
	{
		Printf("No such ban\n");
	}
}

/*
 * Enforce all bans to a specific player. Returns 1 if we banned him
 * and 0 otherwise.
 */
int SV_EnforceBans(int cl)
{
	return (ban_list.enforce_to_player(cl)) ? 1 : 0;
}

/*
 * Re-read all local bans and set the global bans to the ones specified
 * by the "buf" parameter to this function.
 */
void SV_ReReadBans(const char *buf)
{
	char	*tmp;

	/*
	 * Allow this to come only from localhost or from the master
	 */
	if (!is_trusted_address(&net_from))
	{
		Printf("ReReadBans command does not come from master or localhost. Ignored\n");
		return;
	}

	ban_list.flush();
	SV_ReadLocalBans();
	if (buf && *buf && (tmp=strdup(buf)) != NULL )
	{
		readbans_mem(tmp);
		free(tmp);
	}
	ban_list.enforce_to_all();
}


class mem_buf
{
	private:
		int			maxsz, cursz;
		char		*buf;

	public:
					mem_buf();
					~mem_buf();
		void		append(unsigned char *s,int len);
		int			size()			{ return cursz; }
		const char *data()			{ return buf; }
};

mem_buf::mem_buf()
{
	cursz = 0;
	maxsz = 16384;
	if ( (buf=(char *)malloc(maxsz+1)) == NULL )
		maxsz = 0;
}

mem_buf::~mem_buf()
{
	if (buf) free(buf);
}

void mem_buf::append(unsigned char *s,int len)
{
	if (len<=0) return;
	if (cursz+len>maxsz) return;
	memcpy(buf+cursz,s,len);
	cursz += len;
	buf[cursz] = '\0';
}

static int
transfer_banlist(void *cookie,unsigned char *buf,int bufsz)
{
	((mem_buf *) cookie)->append(buf,bufsz);
	return 1;
}

static void STACK_ARGS 
fetch_master_ban_list_aux(void *pv)
{
	int			n;
	mem_buf		mb;

	if (inet_fetch(MASTER_BAN_URL,transfer_banlist,0,0,0,NULL,&mb,60,NULL) != INETERR_NOERROR)
		return;

	if ( (n=mb.size()) > 0)
	{
		// Send a udp message to ourselves to notify the primary thread to 
		// re-read the master banlist
		ZD_OutPacket	ZDOP;	// Use local copy instead of global one
								// because this is in a separate thread
		ZDOP.Init();
		ZDOP.WriteByte(READ_BANLIST);
		ZDOP.WriteString(mb.data());
		ZDOP.SendTo(localhost_addr(),true);
	}
}

void 
SV_FetchMasterBanList(void)
{
	if (master_advertise==0) return;

	#if _WIN32
		_beginthread(fetch_master_ban_list_aux,0,NULL);
	#else
		switch (fork())
		{
			case -1:
				Printf("fork() failed. Cannot fetch master ban list\n");
				break;
			case 0:
				fetch_master_ban_list_aux(NULL);
				_exit(0);
				break;
		}
	#endif
}

void SV_ReFreshBans(void)
{
	unsigned long			cur_time;
	static unsigned long	last_refresh = 0;

	/*
	 * Allow this to come only from the master or localhost
	 */
	if (!is_trusted_address(&net_from))
	{
		Printf("RefreshBans command does not come from master or localhost. Ignored\n");
		return;
	}

	/*
	 * Ensure this does not happen too often (DOS/DDOS precaution)
	 */
	cur_time = time(NULL);
	if (cur_time - last_refresh < 60)
	{
		Printf("Ignoring too frequent ban refresh command.\n");
		return;
	}
	last_refresh = cur_time;
	SV_FetchMasterBanList();
}
